
/** 
  * Module for monitoring Outgoing Traffic, Mobility among the nodes and Channel assesments
  *
  * @author Wasif Masood
  * @date   July 11th, 2010
  */ 


#include "Analyzer.h"
#include "SerialLogger.h"

#ifndef PACKET_VECTOR_SIZE
#define PACKET_VECTOR_SIZE 20
#endif

module AnalyzerP{

	provides interface PacketLoad;
	provides interface RadioAnalyzer;
	
  uses interface Timer<TMilli> as OnTimer;
//  uses interface Timer<TMilli> as MonitorTimer;
	uses interface Timer<TMilli> as RadioTimer;						
	uses interface LocalTime<TMilli>;
	uses interface Packet;

	#if defined(LOG_ANALYZER)
	uses interface SerialLogger;
	#endif
} 
 
implementation{
	
	float alpha = 0.7;
	bool energyLogging = FALSE;	

	traffic_load_msg_t trafficLoad[PACKET_VECTOR_SIZE];
	traffic_load_msg_t totalTrafficLoad;
	analyzer_log_msg_t analyzer_msg;

	struct PacketDelayInfo {
		uint16_t dsn;	
		uint8_t frameType;
		uint16_t pSendTime;
		uint16_t pSendDoneTime;
		error_t error;
	};

/** Control Variables for Packet Delay**/
	struct PacketDelayInfo PacketDelay;

/* Control variables for RadioAnalyzer*/
	uint32_t rStartTime = 0;
	uint32_t rStopTime = 0;
	
	uint32_t rTicOff = 0;
	uint32_t rTocOff = 0;

	uint64_t rOnTime = 0;
	uint64_t rOffTime = 0;

	uint32_t latest = 1;

	analyzer_log_msg_t analyzer_msg;
	energy_log_msg_t energy_msg;

/* Local Functions */
	void refreshTxDataRate();
	void refreshRxDataRate();
	void refreshTxErrorRate();
	void refreshThroughput();	


	/************************************************************************/
	/***************************** RadioAnalyer *****************************/
	/************************************************************************/
	
	command void RadioAnalyzer.logStart() 
	{
		uint32_t diff = 0;		
		rStartTime = call LocalTime.get();

		#ifdef PRINTF_ANALYZER_RADIO
		printf("RadioAnalyzer.logStart() %ld\n", rStartTime);printfflush();
		#endif
		rTocOff = rStartTime;
		
		if(rTicOff!=0){
			diff = rTocOff - rTicOff;
			rOffTime = rOffTime + diff;
		}
			
		energyLogging = TRUE;
		call RadioTimer.startOneShot(100);	
	}

	command void RadioAnalyzer.logStop() {

		uint32_t diff = 0;

		rStopTime = call LocalTime.get();
		rTicOff= rStopTime;

		/** Condition Check for Timer overflow **/
		/*
		if(rStartTime > rStopTime){
			diff = 65534L - rStartTime;
			diff+= rStopTime;
		}
		else
		*/
		
		if(rStartTime!=0)
		{
			diff = rStopTime - rStartTime;
			rOnTime = rOnTime + diff;
		}	  	

		#ifdef PRINTF_ANALYZER_RADIO
			printf("rOnTime: %ld\n", rOnTime);printfflush();
		#endif

		energyLogging = FALSE;
	}

	command uint32_t RadioAnalyzer.getOnTime() {
		
		return rOnTime;
	}

	command uint32_t RadioAnalyzer.getOffTime() {
		
		return rOffTime;
	}

	command void RadioAnalyzer.reset() {
		rOnTime = 0;
		rOffTime = 0;
	}

	command uint64_t RadioAnalyzer.getEnergyConsumed() {
		uint64_t P_on=57,P_off=8;//7.6~8;//I_on=19,I_off=2.55, V=3;
		uint64_t E;

		//P = I*V;
		E = P_on * rOnTime + P_off*rOffTime ;						
		return E;	
	}

	void RadioAnalyzerLogging()
	{
		uint32_t diff = 0;
		uint32_t curTime = 0;
		
		curTime = call LocalTime.get();
		diff = curTime - rStartTime;
		rStartTime = curTime;
		rOnTime = rOnTime + diff;		
	}
	
	event void RadioTimer.fired()
	{
		if(energyLogging)
		{
			RadioAnalyzerLogging();
			call RadioTimer.startOneShot(100);
		}
	}

	/************************************************************************/
	/**************************** PacketLoad *****************************/
	/************************************************************************/

	/*********************** Commands for PacketLoad ***************************/
		/*Monitoring Outbound traffic rate in msec*/
	command void PacketLoad.setMonitoringRate(uint16_t rate){
		call OnTimer.stop();		
		call OnTimer.startPeriodic(rate);				
	}

	/* No. of bytes transmitted / sec*
	command uint16_t PacketLoad.getTxDataRate(uint8_t idx){
		return trafficLoad[idx].sendRate;
	}

	/* No. of packets dropped / sec*
	command uint16_t PacketLoad.getTxDropRate(uint8_t idx) {
		return trafficLoad[idx].dropRate;
	}

	/* No. of bytes received / sec*
	command uint16_t PacketLoad.getRxDataRate(uint8_t idx) {
		return trafficLoad[idx].receiveRate;
	}

	command uint16_t PacketLoad.getThroughputRate(uint8_t idx) {
		return trafficLoad[idx].throughputRate;
	}

	command uint16_t PacketLoad.getTotalTxDataRate(){
		return totalTrafficLoad.sendRate;
	}

	/* No. of packets dropped / sec*
	command uint16_t PacketLoad.getTotalTxDropRate(){
		return totalTrafficLoad.dropRate;
	}

	/* No. of bytes received / sec*
	command uint16_t PacketLoad.getTotalRxDataRate() {
		return totalTrafficLoad.receiveRate;
	}

	command uint16_t PacketLoad.getTotalThroughputRate() {
		return totalTrafficLoad.throughputRate;
	}*/
	
	command void* PacketLoad.getLoad(int8_t idx){
		if(idx>0)
			return &trafficLoad[idx];
		else
			return &totalTrafficLoad;
	}

	command void PacketLoad.logPacketSend(uint8_t pType,uint8_t len) {

		trafficLoad[pType].packetsSend++;
		trafficLoad[pType].send_bytes += len;	

		totalTrafficLoad.packetsSend++;
		totalTrafficLoad.send_bytes += len;

	}

	command void PacketLoad.logPacketSendDone(uint8_t pType,message_t *msg,error_t err) {				

		trafficLoad[pType].packetsSend++;
		totalTrafficLoad.packetsSend++;		

		//printf("trafficLoad[%u].sent_bytes:%u\n",pType,trafficLoad[pType].sent_bytes);printfflush();

		if(err == SUCCESS){
			trafficLoad[pType].sent_bytes += call Packet.payloadLength(msg);
			totalTrafficLoad.sent_bytes += call Packet.payloadLength(msg);
			trafficLoad[pType].send_count++;
		}
		else{
			trafficLoad[pType].packetsLost++;
			totalTrafficLoad.packetsLost++;
		}	
	}		

	command void PacketLoad.logPacketReceive(uint8_t pType,uint8_t len) {
		trafficLoad[pType].receive_bytes += len;
		totalTrafficLoad.receive_bytes += len;
		trafficLoad[pType].receive_count++;
	}
	
	void log_to_serial(int8_t idx){
		
		#ifdef TOSSIM 
		dbg("Analyzer","pType:%u\n",analyzer_msg.pType);
		dbg("Analyzer","rxDataRate:%u\n",analyzer_msg.rxDataRate);
		dbg("Analyzer","throughputRate:%u\n",analyzer_msg.throughputRate);
		dbg("Analyzer","txErrorRate:%u\n",analyzer_msg.txErrorRate);
		#endif

		#ifdef LOG_ANALYZER
		analyzer_msg.pType = idx;
		analyzer_msg.rxDataRate = (nx_uint16_t)trafficLoad[idx].receiveRate;
		analyzer_msg.throughputRate = trafficLoad[idx].throughputRate;
		analyzer_msg.txErrorRate = trafficLoad[idx].txErrorRate;
		call SerialLogger.send(&analyzer_msg,sizeof(analyzer_log_msg_t),AM_ANALYZER_LOG_MSG);
		#endif
	}


	/*********************** End of PacketLoad Commands ***************************/

	/*********************** events of Timer ***************************
		Calculating Exponential weighted moving average every second.
	*/
	event void OnTimer.fired()
	{		
		refreshTxErrorRate();
		refreshRxDataRate();
		refreshThroughput();
		
	#ifdef LOG_ANALYZER
		log_to_serial(BMAC_MSG);
//		log_to_serial(S4_BEACON_MSG);
//		log_to_serial(S4_APP_MSG);
//		log_to_serial(S4_FWD_MSG);
//		log_to_serial(S4_DV_MSG);
		log_to_serial(CTP_BEACON_MSG);
		log_to_serial(CTP_APP_MSG);
		log_to_serial(CTP_FWD_MSG);
		log_to_serial(APP_MSG);
	#endif
		
//		printf("now:%u\n",call OnTimer.getdt());printfflush();

	#ifdef LOG_ENERGY
		energy_msg.energyConsumed =  call RadioAnalyzer.getEnergyConsumed(); 
		call SerialLogger.send(&energy_msg,sizeof(energy_log_msg_t),AM_ENERGY_LOG_MSG);
	#endif
	}

	void refreshThroughput(){
		int i;
		for(i=0;i<PACKET_VECTOR_SIZE;i++){
			trafficLoad[i].throughputRate = trafficLoad[i].throughputRate * alpha + (1- alpha)*trafficLoad[i].sent_bytes;
			trafficLoad[i].sent_bytes=0;
		}
		totalTrafficLoad.throughputRate = totalTrafficLoad.throughputRate * alpha + (1- alpha)*totalTrafficLoad.sent_bytes;
		totalTrafficLoad.sent_bytes=0;
	}

	/*
	void refreshTxDataRate() {	
		int i;		
		for(i=0;i<PACKET_VECTOR_SIZE;i++){
			trafficLoad[i].sendRate = trafficLoad[i].sendRate * alpha + (1-alpha) * trafficLoad[i].send_bytes;
			trafficLoad[i].send_bytes = 0;
		}
		totalTrafficLoad.sendRate = totalTrafficLoad.sendRate * alpha + (1-alpha) * totalTrafficLoad.send_bytes;
		totalTrafficLoad.send_bytes = 0;
	}*/

	void refreshRxDataRate() {	
		int i;
		for(i=0;i<PACKET_VECTOR_SIZE;i++){	
			trafficLoad[i].receiveRate = trafficLoad[i].receiveRate * alpha + (1-alpha) * trafficLoad[i].receive_bytes;
			trafficLoad[i].receive_bytes = 0;		
		}
		totalTrafficLoad.receiveRate = totalTrafficLoad.receiveRate * alpha + (1-alpha) * totalTrafficLoad.receive_bytes;
		totalTrafficLoad.receive_bytes = 0;
	}

/* Calculating weighted average over outbound and inbound packets*/
	void refreshTxErrorRate(){
		int i;
		
		for(i=0;i<PACKET_VECTOR_SIZE;i++){
			if(trafficLoad[i].packetsSend == 0)
				latest = 0;
			else
				latest = 100 * trafficLoad[i].packetsLost / trafficLoad[i].packetsSend;
			trafficLoad[i].txErrorRate = trafficLoad[i].txErrorRate * alpha + (1-alpha) * latest;
			trafficLoad[i].packetsSend = 0;
			trafficLoad[i].packetsLost = 0;
		}
		
		if(totalTrafficLoad.packetsSend == 0)
			latest = 0;
		else 
			latest = 100*totalTrafficLoad.packetsLost/totalTrafficLoad.packetsSend;
		
		totalTrafficLoad.txErrorRate = totalTrafficLoad.txErrorRate * alpha + (1-alpha) * latest;

		totalTrafficLoad.packetsSend = 0;
		totalTrafficLoad.packetsLost = 0;		

	}

	#if defined(LOG_ANALYZER)	
	event void SerialLogger.sendDone(message_t *msg,error_t err){}
	#endif
}




