#include "Python.h"
#include "pythread.h"

static PyThread_type_lock thread_start_event;
static PyThread_type_lock thread_call_event;
static PyThread_type_lock thread_wait_event;
static PyObject *thread_callback;

static PyObject *
call_in_c_thread(PyObject *self, PyObject *args)
{
    PyObject *callback;

    if (!PyArg_ParseTuple(args, "O:call_in_c_thread", &callback))
        return NULL;

    Py_INCREF(callback);
    thread_callback = callback;

    Py_BEGIN_ALLOW_THREADS
        printf("Python thread: notify the C thread that a callback is available\n");
        PyThread_release_lock(thread_call_event);

        printf("Python thread: wait until the C thread called the callback\n");
        PyThread_acquire_lock(thread_wait_event, 1);
        PyThread_release_lock(thread_wait_event);
    Py_END_ALLOW_THREADS

    printf("Python thread: callback has been called, we are done\n");
    Py_RETURN_NONE;
}

static PyMethodDef test_methods[] = {
    {"call_in_c_thread", call_in_c_thread, METH_VARARGS,
     PyDoc_STR("set_error_class(error_class) -> None")},
    {NULL,              NULL}           /* sentinel */
};

PyDoc_STRVAR(module_doc,
"_test module.");

static void
test_thread(void *unused)
{
    PyGILState_STATE state;
    PyObject *res;

    PyThread_release_lock(thread_start_event);

    printf("C thread: wait the callback...\n");
    PyThread_acquire_lock(thread_call_event, 1);
    printf("C thread: got the callback\n");
    PyThread_release_lock(thread_call_event);

    printf("C thread: acquire the GIL\n");
    state = PyGILState_Ensure();

    printf("C thread: call the callback\n");
    res = PyObject_CallFunction(thread_callback, "", NULL);
    Py_DECREF(thread_callback);
    thread_callback = NULL;

    if (res == NULL) {
        printf("C thread: the callback failed!\n");
        PyErr_Print();
    }
    else {
        printf("C thread: the callback succeed\n");
        Py_DECREF(res);
    }

    printf("C thread: release the GIL\n");
    PyGILState_Release(state);

    printf("C thread: callback has been called, notify the Python thread\n");
    PyThread_release_lock(thread_wait_event);

    printf("C thread: exit\n");
}

PyMODINIT_FUNC
init_test(void)
{
    PyObject *m;

    PyEval_InitThreads();

    thread_start_event = PyThread_allocate_lock();
    thread_call_event = PyThread_allocate_lock();
    thread_wait_event = PyThread_allocate_lock();
    if (!thread_start_event || !thread_call_event || !thread_wait_event) {
        PyErr_SetString(PyExc_RuntimeError, "could not allocate lock");
        return;
    }
    PyThread_acquire_lock(thread_start_event, 1);
    PyThread_acquire_lock(thread_call_event, 1);
    PyThread_acquire_lock(thread_wait_event, 1);

    if (PyThread_start_new_thread(test_thread, NULL) == -1) {
        PyErr_SetString(PyExc_RuntimeError, "unable to start the thread");
        return;
    }

    printf("Python thread: wait until the C thread started\n");
    PyThread_acquire_lock(thread_start_event, 1);
    PyThread_release_lock(thread_start_event);
    printf("Python thread: the C thread started\n");

    m = Py_InitModule3("_test", test_methods, module_doc);
    if (m == NULL)
        return;
}

