/*
 * Original author: Steven Feuerstein, 1996 - 2002
 * PostgreSQL implementation author: Pavel Stehule, 2006
 * Further changes by Alvaro Herrera for Command Prompt, Inc and CashNetUSA,
 * 2009
 *
 * This module is under BSD Licence
 */
#include "postgres.h"

#include <stdlib.h>
#include <string.h>

#include "access/heapam.h"
#include "catalog/pg_type.h"
#include "funcapi.h"
#include "utils/builtins.h"
#include "utils/elog.h"
#include "utils/memutils.h"
#include "utils/pg_locale.h"

PG_MODULE_MAGIC;


Datum format_errcxt_stack(PG_FUNCTION_ARGS);
PG_FUNCTION_INFO_V1(format_errcxt_stack);

#define NUM_FNSTACK_ELEM 4

typedef struct ErrstackContext
{
	char   *errcontext;
	int		parsepos;
	int		stackpos;
	bool	finished;
	TupleDesc tupdesc;
} ErrstackContext;

/*
 * Returns the current error context as a newline-separated string of
 * entries, palloc'd in the caller's memory context.
 */
static char *
get_errcontext(void)
{
	ErrorData	   *edata;
	char		   *oldvalue;
	ErrorContextCallback *econtext;
	char		   *retval;

	/* we need the errcontext functions to run in the C locale */
	oldvalue = pstrdup(GetConfigOption("lc_messages"));
	SetConfigOption("lc_messages", "C", PGC_INTERNAL, PGC_S_OVERRIDE);

#if PG_VERSION_NUM >= 80400
	errstart(ERROR, __FILE__, __LINE__, PG_FUNCNAME_MACRO, TEXTDOMAIN);
#else
	errstart(ERROR, __FILE__, __LINE__, PG_FUNCNAME_MACRO);
#endif

	/* now execute the errcontext callbacks to fill errordata */
	for (econtext = error_context_stack;
		 econtext != NULL;
		 econtext = econtext->previous)
		(*econtext->callback) (econtext->arg);

	/* restore the locale setting */
	SetConfigOption("lc_messages", oldvalue, PGC_INTERNAL, PGC_S_OVERRIDE);

	edata = CopyErrorData();

	FlushErrorState();

	if (edata->context)
		retval = pstrdup(edata->context);
	else
		retval = NULL;
	FreeErrorData(edata);

	return retval;
}

Datum
format_errcxt_stack(PG_FUNCTION_ARGS)
{
	FuncCallContext *funcctx;
	MemoryContext	oldcontext;
	char		   *start;
	ErrstackContext *fctx;
	char		   *eol;

	if (SRF_IS_FIRSTCALL())
	{
		char   *context;
		TupleDesc	tupdesc;

		funcctx = SRF_FIRSTCALL_INIT();

		/* Switch context when allocating stuff to be used in later calls */
		oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

		context = get_errcontext();

		/* fast path if there's no context */
		if (context == NULL)
			SRF_RETURN_DONE(funcctx);

		/* Construct a tuple descriptor for the result rows. */
		tupdesc = CreateTemplateTupleDesc(NUM_FNSTACK_ELEM, false);
		TupleDescInitEntry(tupdesc, (AttrNumber) 1, "type",
						   TEXTOID, -1, 0);
		TupleDescInitEntry(tupdesc, (AttrNumber) 2, "name",
						   TEXTOID, -1, 0);
		TupleDescInitEntry(tupdesc, (AttrNumber) 3, "line",
						   INT4OID, -1, 0);
		TupleDescInitEntry(tupdesc, (AttrNumber) 4, "position",
						   INT4OID, -1, 0);

		fctx = palloc(sizeof(ErrstackContext));
		fctx->errcontext = context;
		fctx->parsepos = 0;
		fctx->stackpos = 1;
		fctx->finished = false;
		fctx->tupdesc = BlessTupleDesc(tupdesc);

		funcctx->user_fctx = fctx;
		
		MemoryContextSwitchTo(oldcontext);
	}

	funcctx = SRF_PERCALL_SETUP();
	fctx = funcctx->user_fctx;

	start = fctx->errcontext + fctx->parsepos;

	/* finished parsing the string */
	if (*start == '\0' || fctx->finished)
		SRF_RETURN_DONE(funcctx);

	/*
	 * Turn the first newline into a end-of-string so that the parsing
	 * routine knows where to stop looking.  This also sets the position
	 * where the next loop will start parsing.
	 */
	eol = strchr(start, '\n');
	if (eol)
	{
		*eol = '\0';
		fctx->parsepos = eol - fctx->errcontext + 1;
	}
	else
		fctx->finished = true;

	{
		char   *oname =  "anonymous object";
		char   *line = NULL;
		char   *type = "";
		Datum	values[NUM_FNSTACK_ELEM];
		bool	nulls[NUM_FNSTACK_ELEM];
		HeapTuple tuple;

		/*
		 * lines of the form: PL/pgSQL function "foobar"
		 * plpgsql_exec_error_callback()
		 */
		if (strncmp(start, "PL/pgSQL function ", 18) == 0)
		{
			char   *p1;

			type = "PL/pgSQL function";

			if ((p1 = strstr(start, "function \"")))
			{
				char   *p2;

				p1 += strlen("function \"");

				/* Note that this fails for function name with " in them */
				if ((p2 = strchr(p1, '"')))
				{
					*p2++ = '\0';
					oname = p1;
					start = p2;
				}
			}

			if ((p1 = strstr(start, "line ")))
			{
				int		end;
				char	c;

				p1 += strlen("line ");
				end = strspn(p1, "0123456789");

				c = p1[end];
				p1[end] = '\0';
				line = pstrdup(p1);
				p1[end] = c;

				start = p1 + end;
			}
		}

		/*
		 * lines of the form:
		 * SQL statement in PL/PgSQL function "foo" near line %d
		 * plpgsql_sql_error_callback()
		 */
		if (strncmp(start, "SQL statement in PL/PgSQL function ", 35) == 0)
		{
			char   *p1;

			if ((p1 = strchr(start, '\"')))
			{
				char   *p2;

				if ((p2 = strchr(p1, '\"')))
					*p2 = '\0';

				type = "SQL statement in PL/pgSQL function";
				oname = p1;
			}
		}

		values[0] = PointerGetDatum(DirectFunctionCall1(textin,
														CStringGetDatum(type)));
		nulls[0] = false;

		values[1] = PointerGetDatum(DirectFunctionCall1(textin,
														CStringGetDatum(oname)));
		nulls[1] = false;

		if (line)
		{
			values[2] = Int32GetDatum(DirectFunctionCall1(int4in,
														  CStringGetDatum(line)));
			nulls[2] = false;
			pfree(line);
		}
		else
			nulls[2] = true;

		values[3] = Int32GetDatum(fctx->stackpos++);
		nulls[3] = false;

		/* done with this line */
		tuple = heap_form_tuple(fctx->tupdesc, values, nulls);
		SRF_RETURN_NEXT(funcctx, HeapTupleGetDatum(tuple));
	}

	elog(ERROR, "shouldn't be here at all");

	/* keep compiler quiet */
	SRF_RETURN_DONE(funcctx);
}
