#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <time.h>

#include "tlzo.h"
#include "helpers.h"

#define IFNAME "rtest.c"

/*
 * Because I'm to lazy to rework my BUGCHK macro to be able to
 * handle a version without the __VA_ARGS__ bit some of the bits below
 * might look strange. (ie: BUGCHK( expr, "%s", "failure!" );)
 */

/*
 * I've recently been running into odd returns from the testing. To counter
 * this I've decided to have the compression/decompression tests automatically
 * re-run themselves *before* returning data to the user if the time of the
 * test exceeds 1000. This is to counter these seemingly random blips that have
 * been creating problems, like the script that runs the benchmark returning
 * bad results.
 */

char *base, *comp, *decomp, *work, *rbase, *rcomp, *rdecomp, *rwork;
int setup() {
  size_t len;
  int t, input_file;
  struct stat f_info;

  BUGCHK( (input_file = open( IFNAME, O_RDWR )) == -1,
	  "Unable to open %s: %s", IFNAME, strerror(errno) );
  BUGCHK( lstat( IFNAME, &f_info ) == -1,
	  "fstat( [fd = ]%i, &f_info ) failed: %s",
	  input_file, strerror(errno) );
  len = f_info.st_size;
  len += 3;
 
  base = (char *)malloc( len );
  BUGCHK( base == NULL, "malloc( [size =]%i )", len, strerror(errno) );
  memset( base, 0, len );
  t = read( input_file, base, len );
  BUGCHK( t == -1, "read() failed: %s", strerror(errno) );
  BUGCHK(t<(len-3),"read() failed - was supposed to read %i bytes, read %i",
	  len, t );

  comp = (char *)malloc( len );
  BUGCHK( comp == NULL, "comp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( comp, 0, len ); 
  decomp = (char *)malloc( len );
  BUGCHK( decomp == NULL, "decomp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( decomp, 0, len );
  work = (char *)malloc( LZO1X_WORKMEM_SIZE );
  BUGCHK( work == NULL, "work = (char *)malloc( [len =]%i ) failed: %s",
	  LZO1X_WORKMEM_SIZE, strerror(errno) );
  memset( work, 0, LZO1X_WORKMEM_SIZE );

  /* allocate for the random test */
  rwork = (char *)malloc( LZO1X_WORKMEM_SIZE );
  BUGCHK( rwork == NULL, "rwork = (char *)malloc( [len =]%i ) failed: %s",
	  LZO1X_WORKMEM_SIZE, strerror(errno) );
  memset( rwork, 0, LZO1X_WORKMEM_SIZE );

  /* lets use the same size random data as we used for the baseline test */
  rbase = malloc( len );
  BUGCHK( rbase == NULL, "malloc( [size =]%i )", len, strerror(errno) );
  memset( rbase, 0, len );
  rcomp = (char *)malloc( len );
  BUGCHK( rcomp == NULL, "comp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( rcomp, 0, len ); 
  rdecomp = (char *)malloc( len );
  BUGCHK( rdecomp == NULL, "decomp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( rdecomp, 0, len );
  
  close( input_file );
  return len;

 err_out:
  free( rcomp );
  free( rdecomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  close( input_file );
  return -1;

}

int setup_random( int len ) {
  int fd, rv;
  int rl = (len - 3) / 4;

  BUGCHK( (fd = open( "/dev/urandom", O_RDONLY )) == -1,
	  "Unable to open /dev/urandom for input of random data: %s",
	  strerror(errno) );
  rv = read( fd, rbase, rl );
  BUGCHK( rv == -1, "Error reading /dev/urandom: %s", strerror(errno) );
  BUGCHK( rv < rl, "Error reading /dev/urandom: asked for %i bytes,"\
	  "got %i bytes", rv, len );
  return 0;

 err_out:
  close( fd );
  free( rdecomp );
  free( rcomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  return -1;
}

int rcompress( int len ) {
  size_t rlen, olen;
  int t, k;
  unsigned long long val;
  struct timeval b, a;

 do_test:
  rlen = len - 3;
  olen = rlen;
  
  k = gettimeofday( &b, NULL );
  t = lzo1x_compress( rbase, rlen, rcomp, &olen, work );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != 0,
	  "%s ( t == %i )", "lzo1x_compress has failed", t);

  val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
  if( val > 1000 ) goto do_test; // discard any values above a given threshold
  INFO( "RC: %u", val );

  return olen;

 err_out:
  free( rdecomp );
  free( rcomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  return -1;
}
  
int rdecompress( int size, int orig ) {
  size_t olen;
  int t, k;
  unsigned long long val;
  struct timeval a, b;

 do_test:
  olen = orig;
  k = gettimeofday( &b, NULL );
  t = lzo1x_decompress( rcomp, size, rdecomp, &olen );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != 0,
	  "lzo1x_decompress has failed ( t == %i )", t);

  val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
  if( val > 1000 ) goto do_test;
  INFO( "RD: %u", val );

  return olen;

 err_out:
  free( rdecomp );
  free( rcomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  return -1;
}
  
int compress( int len ) {
  size_t rlen, olen;
  int t, k;
  unsigned long long val;
  struct timeval b, a;

do_test:
  rlen = len - 3;
  olen = rlen;
  
  k = gettimeofday( &b, NULL );
  t = lzo1x_compress( base, rlen, comp, &olen, work );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != 0,
	  "%s ( t == %i )", "lzo1x_compress has failed", t);
  val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
  if( val > 1000 ) goto do_test;
  INFO( "C: %u", val );

  return olen;

 err_out:
  free( rdecomp );
  free( rcomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  return -1;
}

int decompress( int size, int orig ) {
  size_t olen;
  int err = 0, t, k;
  unsigned long long val;
  struct timeval a, b;

 do_test:
  olen = orig;
  k = gettimeofday( &b, NULL );
  t = lzo1x_decompress( comp, size, decomp, &olen );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != 0,
	  "%s ( t == %i )", "lzo1x_decompress has failed", t);

  val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
  if( val > 1000 ) goto do_test;
  INFO( "D: %u", val );

  return olen;

 err_out:
  free( rdecomp );
  free( rcomp );
  free( rbase );
  free( rwork );
  free( work );
  free( decomp );
  free( comp );
  free( base );
  return -1;
}

int main() {
  int clen, dlen, len, k;

  len = setup();
  BUGCHK( len <= 0, "setup() failed - result code: %i", len );
  k = setup_random(len);
  BUGCHK( k < 0,
	  "setup_random(%i) failed: %i", len, k );

  clen = compress( len );
  BUGCHK( clen == -1, "%s", "Compression routine failure" );
  dlen = decompress( clen, len );
  BUGCHK( dlen == -1, "%s", "Decompression routine failure" );

  clen = rcompress( len );
  BUGCHK( clen == -1, "%s", "Random data compression failure" );
  dlen = rdecompress( clen, len );
  BUGCHK( dlen == -1, "%s", "Random data decompression failure" );

  return 0;

 err_out:
  return 1;
}
