-------------------------------------------------------------------------------
-- $Id: Tile.hs,v 1.10 2000/01/12 08:18:04 satnam Exp $
-------------------------------------------------------------------------------

module Tile
where
import Gates
import TypeExpr
import Utils
import Array
import Utils

-- This module includes an experiment for layout out Lava tiles.

-------------------------------------------------------------------------------

type Pos = (Int, Int)

-- A position is either unspecified or an (x,y) coordinate with
-- (0,0) in the bottom left-hand corner [N.B. not the same as
-- the RLOC location system].

data Origin
  = NoOrigin | Origin Int Int
    deriving (Eq, Show)

data Position
  = Position Pos
  | USet String Pos Origin
  | Unplaced 
    deriving (Eq, Show)
    

                        
-------------------------------------------------------------------------------

type InstanceArray = Array Int InstanceTuple             
            
-------------------------------------------------------------------------------

type InstanceTuple = (Cell,                -- Cell name
                      Position,            -- Position
                      [Property],          -- Attributes
                      [PortMap]            -- Port connections and modes
                     )
                      
-------------------------------------------------------------------------------

showInstanceTuple :: InstanceTuple -> String
showInstanceTuple (Unused, _, _, _) = "- " 
showInstanceTuple (cell, pos, props, ports)
  = show cell ++ " " ++ showPorts ports ++ " " ++
    showPlacement pos ++ showProps props

-------------------------------------------------------------------------------

showPorts ports
  = "(" ++ insertString ", " (map showPort ports) ++ ")"
  
  
showPort (PortMap mode arg par)
  = showMode mode ++ " " ++  arg ++ " => " ++ showSignal par
  
showMode Input = "in"
showMode Output = "out"
 
showPlacement Unplaced = ""
showPlacement (Position pos) = show pos
showPlacement (USet uset pos origin)
  = "uset " ++ show uset ++ " " ++ show pos ++ "origin=" ++ show origin 

-------------------------------------------------------------------------------


showProps [] = ""
showProps props = " " ++ insertString " " (map showProp props)

-------------------------------------------------------------------------------

showProp (InitialBV _ i) = "INIT=" ++ show i
   
-------------------------------------------------------------------------------

showInstanceTupleArray :: Int ->  Array Int InstanceTuple -> String
showInstanceTupleArray n a
  = unlines [show i ++ " : " ++ showInstanceTuple (a!i) | i <- [0..n]]

-------------------------------------------------------------------------------

   
type Tile = ((Int,Int), [Int])

-------------------------------------------------------------------------------

getx pos
  = case pos of
      Position (x,y) -> x
      USet _ (x,y) _ -> x

-------------------------------------------------------------------------------
      
gety pos
  = case pos of
      Position (x,y) -> y
      USet _ (x,y) _ -> y

-------------------------------------------------------------------------------


getxFromGate (cell, inst, pos, prop, ios) = getx pos
getyFromGate (cell, inst, pos, prop, ios) = gety pos


-------------------------------------------------------------------------------

getxyFromGate (cell, inst, pos, prop, ios) = (getx pos, gety pos)

-------------------------------------------------------------------------------

translatePos (dx,dy) Unplaced = Unplaced
translatePos (dx,dy) (Position (x,y)) = Position (x+dx, y+dy)

-------------------------------------------------------------------------------

data Property = InitialBV Int Int
                deriving (Eq, Show)

-------------------------------------------------------------------------------

getInitValue :: [Property] -> Int
getInitValue props
  = if vs == [] then
      error ("Initial value not specified.\n")
    else
      if length vs > 1 then
        error "More than one initial value specified.\n"
      else
        head vs
    where
    vs = [v | InitialBV s v <- props]
    
-------------------------------------------------------------------------------

cellProperties gate
  = case gate of
      Lut1 iv -> [InitialBV  2 iv]
      Lut2 iv -> [InitialBV  4 iv]
      Lut3 iv -> [InitialBV  8 iv]
      Lut4 iv -> [InitialBV 16 iv]
      Rom16x1 iv -> [InitialBV 16 iv]
      other -> []
      
-------------------------------------------------------------------------------

mergeTiles :: [Tile] -> Tile
mergeTiles tiles
  = concatTiles tiles

-------------------------------------------------------------------------------

concatTiles :: [Tile] -> Tile
concatTiles tiles
  = ((maximum (0:xs), maximum (0:ys)), concat insts) 
    where
    tile_xy = map fst tiles
    xs = map fst tile_xy
    ys = map snd tile_xy
    insts = map snd tiles

-------------------------------------------------------------------------------
                