-------------------------------------------------------------------------------
-- $Id: TestSplitMultiplier.hs,v 1.10 2000/01/20 20:14:08 satnam Exp $
-------------------------------------------------------------------------------

module TestSplitMultiplier (splitMultiplierRegressionTests)
where
import Lava
import Virtex
import VirtexLib
import SplitMultiplier
import MotorolaKCM
import KCMUtils
import Testing


-------------------------------------------------------------------------------

-- Some sample data-sets.

-- 4-bit data sets

unsigned_dataset n = [0..2^n-1]
signed_dataset n = [-2^(n-1)..2^(n-1)-1]



-------------------------------------------------------------------------------

simulation_unsigned_result circuit input_size input
  = stream2int resultBits 
    where
    inputBits = intStream input_size input
    resultBits = circuit inputBits

-------------------------------------------------------------------------------

simulation_signed_result circuit input_size input
  = stream2signedint resultBits 
    where
    inputBits = intStream input_size input
    resultBits = circuit inputBits

-------------------------------------------------------------------------------

k  = [0, 1, 2, 3, 4,  50, 51, 52, 60, 61, 62, 63]
sk = [0, 1, 2, 3, 4, 31, -32, -30, -20, -10, -1]


k12 = [4095, 0, 1000, 2000, 3000]

-------------------------------------------------------------------------------

tg n
  = do r <- sequence (map sequence (tgroup n))
       let f = sum (concat r)
       putStrLn ("Failures: " ++ show f)

-------------------------------------------------------------------------------


-- Test group 1

splitMultiplierRegressionTests :: IO ()
splitMultiplierRegressionTests = testSplitMult xtgroups  

-------------------------------------------------------------------------------

testSplitMult :: [TestTuple] -> IO ()
testSplitMult vecs
  = do r <- sequence [elaborate_and_test_kcm t | t <- vecs] 
       putStrLn ("Failures: " ++ show (sum r) ++ " out of " ++
                   show (length r) ++ " tests.")
                   
-------------------------------------------------------------------------------


smt = splitMultiplierRegressionTests

xtgroups = concat (concat [xtgroup i | i <- [1..5]])

type TestTuple = (Integer, Int, Bool, Int, [Integer])
type TestVecs = [[TestTuple]]

xtgroup ::  Int -> TestVecs 

-- Unsigned 6-bit coief.
-- Unsigned 1-4 bit data
xtgroup 1
  = [[(v, 6, False, i, unsigned_dataset i)  | i <- [1..4], v <- k]]

-- Unsigned 6-bit coief.
-- Unsigned 5-12 bit data
xtgroup 2
  = [[(v, 6, False, i, unsigned_dataset i)  | i <- [5..12], v <- k]]

-- Unsigned 6-bit coief.
-- Signed   1-12 bit data
xtgroup 3
  = [[(v, 6, True, i, signed_dataset i)  | i <- [1..12], v <- k]]

-- Signed    6-bit coief.
-- Unsigned  1-12 bit data
xtgroup 4
  = [[(v, 6, False, i, unsigned_dataset i)  | i <- [1..12], v <- sk]]


-- Signed  6-bit coief.
-- Signed  1-12 bit data
xtgroup 5
  = [[(v, 6, True, i, signed_dataset i)  | i <- [1..12], v <- sk]]

-------------------------------------------------------------------------------

-- Unsigned  12-bit coief.
-- Unsigned  12 bit data
xtgroup 7
  = [[(v, 12, False, i, unsigned_dataset i)  | i <- [1..12], v <- k12]]

-------------------------------------------------------------------------------
-- Data-sheet comparisons
-------------------------------------------------------------------------------

sheet
  = testSplitMult (concat (concat [datasheet i | i <- [1..1]]))

-------------------------------------------------------------------------------

datasheet 1
  = [
     [(32767, 16, False, 1, unsigned_dataset 1)],
     [(65535, 16, False, 1, unsigned_dataset 1)],
     [(15, 4, False, 4, unsigned_dataset 4)],  -- 4x4
     [(15, 4, False, 5, unsigned_dataset 5)],  -- 4x5
     [(15, 4, False, 10, unsigned_dataset 10)], -- 4x10
     [(4095, 12, False, 12, unsigned_dataset 12)]
    ]

-------------------------------------------------------------------------------


xxx :: TestTuple
xxx
  = (-32, 6, False, 5, [1])

t = elaborate_and_test_kcm xxx 

-------------------------------------------------------------------------------

    
   
    
elaborate_and_test_kcm :: Integral num => 
                          (num, Int, Bool, Int, [num]) -> IO Int 
elaborate_and_test_kcm (k, kSize, dataSigned, dataSize, inputs)
  = do circuit2VHDL filename netCir
       r <- checkResult filename expectedResult actualResult
       return r 
    where
    netCir :: StateST state ()
    netCir = kcm_top dataSigned k kSize dataSize
    circuit :: Circuit nsi bit => [bit] -> nsi [bit]  
    circuit = splitKCM dataSigned k kSize 
    filename = "kcm_" ++ showK k ++ showSign (k < 0) ++ "_" ++
                show dataSize ++ showSign dataSigned
    showK n = if n < 0 then
               "minus_" ++ show (abs n)
              else
                show n 
    showSign signed = if signed then
                        "s"
                      else
                        "u"
    expectedResult = pipeDelays ++  [k * i | i <- inputs]
    actualResult = str2int (motkcm dataSigned k kSize) 
                               dataSize inputs  
    signedResult = (k < 0) || dataSigned
    str2int = if signedResult then
                simulation_signed_result
              else
                simulation_unsigned_result
    test_data True = signed_dataset
    test_data False = unsigned_dataset
    pipeDelays = replicate (latency dataSize) 0




tgroup 1       
 =  [ [checkResult ("kcm_6u_4u_sim " ++ show v)
                   (kcm_6u_4u_exp v) (kcm_6u_4u_sim v)  | v <- k],
      [checkResult ("kcm_6s_4u_sim " ++ show v)
                   (kcm_6s_4u_exp v) (kcm_6s_4u_sim v)  | v <-  sk]
    ]

-------------------------------------------------------------------------------

-- Test group 2, usigned coief, [1,2,3]-bit unsigned data
-- Should result in just one lookup table.

tgroup 2
  =  [ [checkResult ("kcm_6u_1u_sim " ++ show v)
                   (kcm_6u_1u_exp v) (kcm_6u_1u_sim v)  | v <- k], 
       [checkResult ("kcm_6u_2u_sim " ++ show v)
                   (kcm_6u_2u_exp v) (kcm_6u_2u_sim v)  | v <- k], 
       [checkResult ("kcm_6u_3u_sim " ++ show v)
                   (kcm_6u_3u_exp v) (kcm_6u_3u_sim v)  | v <- k]
     ]

-------------------------------------------------------------------------------

-- Test group 3, unsigned coief, [1,2,3]-bit signed data
-- Should result in one lookup table but with sign extension.

tgroup 3
  = [ [checkResult ("kcm_6u_1s_sim " ++ show v)
                   (kcm_6u_1s_exp v) (kcm_6u_1s_sim v)  | v <- k],
      [checkResult ("kcm_6u_2s_sim " ++ show v)
                   (kcm_6u_2s_exp v) (kcm_6u_2s_sim v)  | v <- k], 
      [checkResult ("kcm_6u_3s_sim " ++ show v)
                   (kcm_6u_3s_exp v) (kcm_6u_3s_sim v)  | v <- k]
    ]
                   
-------------------------------------------------------------------------------

-- Test group 4, unsigned coief & data, data size in [5..8]

tgroup 4
  = [ [checkResult ("kcm_6u_5u_sim " ++ show v)
                   (kcm_6u_5u_exp v) (kcm_6u_5u_sim v)  | v <- k], 
      [checkResult ("kcm_6u_6u_sim " ++ show v)
                   (kcm_6u_6u_exp v) (kcm_6u_6u_sim v)  | v <- k],
      [checkResult ("kcm_6u_7u_sim " ++ show v)
                   (kcm_6u_7u_exp v) (kcm_6u_7u_sim v)  | v <- k],
      [checkResult ("kcm_6u_8u_sim " ++ show v)
                   (kcm_6u_8u_exp v) (kcm_6u_8u_sim v)  | v <- k]
    ]

-------------------------------------------------------------------------------

-- Test group 5, unsigned coief, signed data, data size in [5..8]

tgroup 5
  = [ [checkResult ("kcm_6u_5s_sim " ++ show v)
                   (kcm_6u_5s_exp v) (kcm_6u_5s_sim v)  | v <- k], 
      [checkResult ("kcm_6u_6s_sim " ++ show v)
                   (kcm_6u_6s_exp v) (kcm_6u_6s_sim v)  | v <- k],
      [checkResult ("kcm_6u_7s_sim " ++ show v)
                   (kcm_6u_7s_exp v) (kcm_6u_7s_sim v)  | v <- k],
      [checkResult ("kcm_6u_8s_sim " ++ show v)
                   (kcm_6u_8s_exp v) (kcm_6u_8s_sim v)  | v <- k]
    ]
                                                
-------------------------------------------------------------------------------

-- Test group 6, signed coief, unsigned data size [1..3]

tgroup 6
  = [ [checkResult ("kcm_6s_1u_sim " ++ show v)
                   (kcm_6u_1u_exp v) (kcm_6s_1u_sim v)  | v <- sk],
      [checkResult ("kcm_6s_2u_sim " ++ show v)
                   (kcm_6s_2u_exp v) (kcm_6s_2u_sim v)  | v <- sk], 
      [checkResult ("kcm_6s_3u_sim " ++ show v)
                   (kcm_6s_3u_exp v) (kcm_6s_3u_sim v)  | v <- sk]
    ]

-------------------------------------------------------------------------------

-- Test group 7, signed coief, signed data size [1..4]

tgroup 7
  = [ [checkResult ("kcm_6s_1s_sim " ++ show v)
                   (kcm_6u_1s_exp v) (kcm_6s_1s_sim v)  | v <- sk],
      [checkResult ("kcm_6s_2s_sim " ++ show v)
                   (kcm_6s_2s_exp v) (kcm_6s_2s_sim v)  | v <- sk], 
      [checkResult ("kcm_6s_3s_sim " ++ show v)
                   (kcm_6s_3s_exp v) (kcm_6s_3s_sim v)  | v <- sk],
      [checkResult ("kcm_6s_4s_sim " ++ show v)
                   (kcm_6s_4s_exp v) (kcm_6s_4s_sim v)  | v <- sk]
    ]

-------------------------------------------------------------------------------


-- A pipelined split-multiplier that takes unsigned data.

kcm_u :: (Integral num, Circuit nsi bit) => 
         num -> Int -> TwosComplement bit ->
         nsi (TwosComplement bit)
kcm_u = splitKCM False

-------------------------------------------------------------------------------

-- A pipelined split-multiplier that takes signed data.

kcm_s :: (Integral num, Circuit nsi bit) => 
         num -> Int -> TwosComplement bit ->
         nsi (TwosComplement bit)
kcm_s = splitKCM True

-------------------------------------------------------------------------------

-- A 6-bit unsigned coiefficent with 4-bit unsigned data input.
-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 4-bit 

kcm_6u_4u_sim v = simulation_unsigned_result (kcm_u v 6) 4 (unsigned_dataset 4)
kcm_6u_4u_exp v = 0 : [v * i | i <- unsigned_dataset 4] 

-------------------------------------------------------------------------------

-- A 6-bit signed coiefficent with 4-bit unsigned data input.
-- COIEF : SIGNED 6-bit
-- DATA  : UNSIGNED 4-bit 

kcm_6s_4u_sim v = simulation_signed_result
                  (kcm_u v 6) 4 (unsigned_dataset 4)
kcm_6s_4u_exp v = 0 : [v * i | i <- unsigned_dataset 4] 



-------------------------------------------------------------------------------
-- Now test using a data input < 4 bits.
-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 1-bit 

kcm_6u_1u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 1 (unsigned_dataset 1)
kcm_6u_1u_exp v = 0 : [v * i | i <- unsigned_dataset 1] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 2-bit 
 
kcm_6u_2u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 2 (unsigned_dataset 2)
kcm_6u_2u_exp v = 0 : [v * i | i <- unsigned_dataset 2] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 3-bit 
 
kcm_6u_3u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 3 (unsigned_dataset 3)
kcm_6u_3u_exp v = 0 : [v * i | i <- unsigned_dataset 3] 

-------------------------------------------------------------------------------
-- And again with signed data.
-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 1-bit 

kcm_6u_1s_sim v = simulation_signed_result
                  (kcm_s v 6) 1 (signed_dataset 1)
kcm_6u_1s_exp v = 0 : [v * i | i <- signed_dataset 1] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 2-bit 

kcm_6u_2s_sim v = simulation_signed_result
                  (kcm_s v 6) 2 (signed_dataset 2)
kcm_6u_2s_exp v = 0 : [v * i | i <- signed_dataset 2] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 3-bit 

kcm_6u_3s_sim v = simulation_signed_result
                  (kcm_s v 6) 3 (signed_dataset 3)
kcm_6u_3s_exp v = 0 : [v * i | i <- signed_dataset 3] 

-------------------------------------------------------------------------------

-- Now test for data size > 4
-- This requires an adder tree containing just one adder
-- so one extra pipelined delay is incurred making the
-- total latency 2 ticks.

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 5-bit 

kcm_6u_5u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 5 (unsigned_dataset 5)
kcm_6u_5u_exp v = 0 : 0 : [v * i | i <- unsigned_dataset 5] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 6-bit 

kcm_6u_6u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 6 (unsigned_dataset 6)
kcm_6u_6u_exp v = 0 : 0 : [v * i | i <- unsigned_dataset 6] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 7-bit 

kcm_6u_7u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 7 (unsigned_dataset 7)
kcm_6u_7u_exp v = 0 : 0 : [v * i | i <- unsigned_dataset 7] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : UNSIGNED 8-bit 

kcm_6u_8u_sim v = simulation_unsigned_result
                  (kcm_u v 6) 8 (unsigned_dataset 8)
kcm_6u_8u_exp v = 0 : 0 : [v * i | i <- unsigned_dataset 8] 

-------------------------------------------------------------------------------

-- Test unsigned data sizes from 5 to 8 bits with
-- unsigned coief.

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 5-bit 

kcm_6u_5s_sim v = simulation_signed_result
                  (kcm_s v 6) 5 (signed_dataset 5)
kcm_6u_5s_exp v = 0 : 0 : [v * i | i <- signed_dataset 5] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 6-bit 

kcm_6u_6s_sim v = simulation_signed_result
                  (kcm_s v 6) 6 (signed_dataset 6)
kcm_6u_6s_exp v = 0 : 0 : [v * i | i <- signed_dataset 6] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 7-bit 

kcm_6u_7s_sim v = simulation_signed_result
                  (kcm_s v 6) 7 (signed_dataset 7)
kcm_6u_7s_exp v = 0 : 0 : [v * i | i <- signed_dataset 7] 

-------------------------------------------------------------------------------

-- COIEF : UNSIGNED 6-bit 
-- DATA  : SIGNED 8-bit 

kcm_6u_8s_sim v = simulation_signed_result
                  (kcm_s v 6) 8 (signed_dataset 8)
kcm_6u_8s_exp v = 0 : 0 : [v * i | i <- signed_dataset 8] 

-------------------------------------------------------------------------------

-- Tests with signed coiefficient.

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : UNSIGNED 1-bit 

kcm_6s_1u_sim v = simulation_signed_result
                  (kcm_u v 6) 1 (unsigned_dataset 1)
kcm_6s_1u_exp v = 0 : [v * i | i <- unsigned_dataset 1] 

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : UNSIGNED 2-bit 
 
kcm_6s_2u_sim v = simulation_signed_result
                  (kcm_u v 6) 2 (unsigned_dataset 2)
kcm_6s_2u_exp v = 0 : [v * i | i <- unsigned_dataset 2] 

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : UNSIGNED 3-bit 
 
kcm_6s_3u_sim v = simulation_signed_result
                  (kcm_u v 6) 3 (unsigned_dataset 3)
kcm_6s_3u_exp v = 0 : [v * i | i <- unsigned_dataset 3] 

-------------------------------------------------------------------------------

-- Test signed coief and signed data size [1..4]

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : SIGNED 1-bit 

kcm_6s_1s_sim v = simulation_signed_result
                  (kcm_s v 6) 1 (signed_dataset 1)
kcm_6s_1s_exp v = 0 : [v * i | i <- signed_dataset 1] 

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : SIGNED 2-bit 
 
kcm_6s_2s_sim v = simulation_signed_result
                  (kcm_s v 6) 2 (signed_dataset 2)
kcm_6s_2s_exp v = 0 : [v * i | i <- signed_dataset 2] 

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : SIGNED 3-bit 
 
kcm_6s_3s_sim v = simulation_signed_result
                  (kcm_s v 6) 3 (signed_dataset 3)
kcm_6s_3s_exp v = 0 : [v * i | i <- signed_dataset 3] 

-------------------------------------------------------------------------------

-- COIEF : SIGNED 6-bit 
-- DATA  : SIGNED 4-bit 
 
kcm_6s_4s_sim v = simulation_signed_result
                  (kcm_s v 6) 4 (signed_dataset 4)
kcm_6s_4s_exp v = 0 : [v * i | i <- signed_dataset 4] 

-------------------------------------------------------------------------------

comb_splitmult :: (Integral num, Circuit nsi bit) =>
                   Bool -> num -> Int -> [bit] -> 
                  nsi [bit]
comb_splitmult = combinationalKCM

pipe_splitmult :: (Integral num, Circuit nsi bit) =>
                   Bool -> num -> Int -> [bit] -> 
                   nsi [bit]
pipe_splitmult = splitKCM


-------------------------------------------------------------------------------

r1=kcm11_6_50_sim1
r2=kcm11_6_50_sim2
r3=kcm11_6_50_sim3
r4=kcm11_6_50_sim4

-- Test the combinational split multiplier.

kcm11_6_50_sim1
  = dispunsigned products -- stream2int products
    where
    products = comb_splitmult False 50 6 input
    input = intStream 11 [1,2,100,200,400,800,1000, 1023] 

-- Expected output: [0,50,100,5000,10000,20000,40000,50000,51150]    
    
-- And now the pipelined version:

kcm11_6_50_sim2
  = stream2int products
    where
    products = pipe_splitmult False 50 6 input
    input = intStream 11 [1,2,100,200,400,800,1000, 1023] 
-- Expected output: [0,0,0,50,100,5000,10000,20000,40000,50000,51150]    


kcm11_6_50_sim3
  = dispsigned products -- stream2signedint products
    where
    products = pipe_splitmult True 30 6 inputs
    inputs = intStream 11 [-1,-2,-3,1,2,3,100,-200,400,-800,500, -512] 
-- Expected output: [-50,100,-5000,10000,-20000,40000,-25000,25600]    


kcm11_6_50_sim4
  = stream2signedint products
    where
    products = pipe_splitmult True (-50) 6 inputs
    inputs = intStream 11 [-1,-2,100,-200,400,-800,500, -512] 
-- Expected output: [-50,100,-5000,10000,-20000,40000,-25000,25600] 

kcm11_6_50Top
  = do addr <- inputVec "a" (bit_vector 10 downto 0)
       product <- comb_splitmult False 50 6 addr
       outputVec product "p" (bit_vector 16 downto 0) 


kcm11_6_50nl = circuit2VHDL "kcm11_6_50" kcm11_6_50Top


-- And the pipelined version
kcm11_6_50_pipeTop
  = do addr <- inputVec "a" (bit_vector 10 downto 0)
       clk <- globalClock "clk"
       clr <- globalClear "clr"
       product <- pipe_splitmult False 50 6 addr
       outputVec product "p" (bit_vector 16 downto 0)


kcm11_6_50_pipenl = circuit2VHDL "kcm11_6_50_pipe" kcm11_6_50_pipeTop

kcm11_6_50_pipenl2 = showNetListToFile "kcm11_6_50_pipe" kcm11_6_50_pipeTop

-- Expected output: [0,50,100,5000,10000,20000,40000,50000,51150]    


k1=kcm4_6_50_sim1
k2=kcm4_6_50_sim2
k3=kcm4_6_50_sim3
k4=kcm4_6_50_sim4
k5=kcm5_6_50_sim5
k6=kcm5_6_50_sim6
k7=kcm5_6_50_sim7
k8=kcm5_6_50_sim8


kcm4_6_50_sim1
  = dispunsigned products -- stream2int products
    where
    products = comb_splitmult False 50 6 input
    input = intStream 4 (reverse [0..15]) 
--Expected: [0,0,50,100,150,200,250,300,350,400,450,500,550,600,650,700,750] 

kcm4_6_50_sim2
  = dispsigned products -- stream2signedint products
    where
    products = comb_splitmult True 50 6 input
    input = intStream 4 [-8..7]
-- Expected: [0,-400,-350,-300,-250,-200,-150,-100,-50,0,50,100,150,200,250,300,350] 

kcm4_6_50_sim3
  = dispsigned products -- stream2signedint products
    where
    products = comb_splitmult False (-50) 7 input
    input = intStream 4 [0..15]    

kcm4_6_50_sim4
  = stream2signedint products
    where
    products = comb_splitmult True (-50) 6 input
    input = intStream 4 [-8..7]


kcm5_6_50_sim5
  = stream2int products
    where
    products = comb_splitmult False 50 6 input
    input = intStream 5 [0..31] 

kcm5_6_50_sim6
  = stream2signedint products
    where
    products = comb_splitmult True 50 6 input
    input = intStream 5 [-16..15]

kcm5_6_50_sim7
  = dispsigned products -- stream2signedint products
    where
    products = comb_splitmult False (-50) 6 input
    input = intStream 5 [0..31]    

kcm5_6_50_sim8
  = stream2signedint products
    where
    products = comb_splitmult True (-50) 6 input
    input = intStream 5 [-16..15]        

-------------------------------------------------------------------------------
