-------------------------------------------------------------------------------
-- $Id: SimulationTests.hs,v 1.26 2000/01/21 06:59:50 satnam Exp $
-------------------------------------------------------------------------------

module SimulationTests
where

import Lava
import Virtex
import VirtexLib
import LazyST

-------------------------------------------------------------------------------

sim1 :: Sim [Bool]
sim1 = inv [False, True]

sim2 :: Sim [Bool]
sim2 = inv [True,False]

-------------------------------------------------------------------------------

nandGate :: Circuit nsi bit => 
            (bit, bit) -> nsi (bit)
nandGate = and2 /\ inv

sim2a :: Sim [Bool]
sim2a = nandGate ([True,True],[True,False])


sym2a :: IO ()
sym2a = runSymbolic (nandGate (Variable "a", Variable "b"))

nandCir :: StateST s (SignalA String)
nandCir = nandGate (Var "a",Var "b") 

nl2 = putStr (showInstanceTupleArray 6 (runST (runSTCir nandCir)))

nandTop :: StateST state ()
nandTop
  = do a <- inputBit "a" BitType
       b <- inputBit "b" BitType
       c <- nandGate (a, b)
       outputBit c "c"

-------------------------------------------------------------------------------

toggleCore :: Circuit nsi bit => 
              (bit, bit) -> nsi (bit, bit) 
toggleCore = and2 >-> inv >-> fork2

-------------------------------------------------------------------------------

sym2b = runSymbolic (toggleCore (Variable "a", Variable "b"))

-------------------------------------------------------------------------------

delayedToggleCore :: Circuit nsi bit => 
                     (bit, bit) -> 
                     nsi (bit, bit) 
delayedToggleCore = and2 >-> inv >-> delayBit False >-> fork2

-------------------------------------------------------------------------------

toggle :: Circuit nsi bit => bit -> nsi (bit)
toggle = loop delayedToggleCore 

-------------------------------------------------------------------------------

sim3 :: Sim [Bool]
sim3 = toggle (replicate 6 True)

sym3 = runSymbolic (toggle (Variable "i"))

-------------------------------------------------------------------------------

toggle2Core :: Circuit nsi bit => 
               bit -> bit -> (bit, bit) -> 
               nsi (bit, bit)
toggle2Core clk clr = and2 >-> inv >-> fdc clk clr >-> fork2               
                
toggle2 :: Circuit nsi bit => 
           bit -> bit -> bit -> nsi (bit) 
toggle2 clk clr = loop (toggle2Core clk clr)

tog2sim :: Sim [Bool]
tog2sim = (toggle2 clk) (replicate 12 True) (True:(repeat False)) 

tog2nl = showNetListB (toggle2 (Var "clk") (Var "clr") (Var "cntl"))

tog2corenl = showNetList (toggle2Core (Var "clk") (Var "clr") (Var "cntl", Var "prev"))

tog2Top
  = do cntl <- inputBit "cntl" BitType
       clk  <- inputBit "clk" BitType
       clr <- inputBit "clr" BitType
       t <- toggle2 clk clr cntl
       outputBit t "t" 

-------------------------------------------------------------------------------

add :: Circuit nsi bit =>
       ([bit], [bit]) -> nsi [bit]
add = ripAdd

-- Circuit: injL 1

counterCore1 :: Circuit nsi bit => 
                [bit] -> nsi ([bit], [bit])     
counterCore1 = injectPair (intConstant 8 1)

-------------------------------------------------------------------------------

-- Circuit: +

counterCore2 :: Circuit nsi bit => 
                ([bit], [bit]) -> nsi [bit]     
counterCore2 = add

-------------------------------------------------------------------------------

counterCore3 :: Circuit nsi bit => 
                [bit] -> nsi [bit]     
counterCore3 = counterCore1 >-> counterCore2

-------------------------------------------------------------------------------

-- Circuit: (+ 1)

counterCore :: Circuit nsi bit => 
               [bit] -> nsi [bit]     
counterCore = injectPair (intConstant 8 1) >-> add

-------------------------------------------------------------------------------

-- sim4 tests a circuit that just adds 1 to its input:

sim4 :: Sim [[Bool]]
sim4 = counterCore [i2bv 8 i | i <- [0..10]]

isim4 = map bv2i (runSim sim4) 

-- Expected output:
-- Main> isim4
-- [1,2,3,4,5,6,7,8,9,10,11]  

-------------------------------------------------------------------------------

-- Circuit: (+ 1) ; delay

counterCore6 :: (Show (nsi [bit]), Circuit nsi bit) => 
                [bit] -> nsi [bit]     
counterCore6 = injectPair (intConstant 8 1) >-> add >-> delayBus 8

-------------------------------------------------------------------------------

-- sim6 is like sim4 except delayed by one clock tick

sim6 :: Sim [[Bool]]
sim6 = counterCore6 (nrList 0 10)

isim6 = stream2int sim6 
isym6 = runSymbolic 
        (counterCore [Variable ("i"++show i) | i <- [0..7]])

-- Expected output:
-- Main> isim6
-- [0,1,2,3,4,5,6,7,8,9,10,11]  

-------------------------------------------------------------------------------

counterCore4 :: Circuit nsi bit => 
                ([bit], [bit]) ->
                nsi ([bit], [bit])     
counterCore4 = add >-> delayBus 8 >-> fork2

-- To check that this counter core is lazy enough for use in
-- a loop the following function should evaluate to a 8-bit 
-- vector of all zeros. If it is undefined then the definition is
-- not lazy enough.

lt4 :: Sim [Bool]
lt4 = headStream (Sim f)
      where
      Sim (v,f) = counterCore4 ([replicate 8 True], undefined)

-------------------------------------------------------------------------------

-- Circuit: loop (+ ; delay ; fork)

counter:: Circuit nsi bit => [bit] -> nsi [bit]
counter = loopList 8 counterCore4

-------------------------------------------------------------------------------

zeros :: [[Bool]]
zeros = [i2bv 8 0 | i <- [0..10]]

ones :: [[Bool]]
ones = (transpose [i2bv 8 1 | i <- [0..100]])


sequ :: [[Bool]]
sequ = [i2bv 8 i | i <- reverse [0..100]]

-------------------------------------------------------------------------------

-- Simulation of a counter.

sim5 :: Sim [[Bool]]
sim5 = takeStream 10 (counter ones)
isim5 = stream2int sim5 

sym5 = runSymbolic 
        (counter [Variable ("i"++show i) | i <- [0..7]])
 


counter5b:: Circuit nsi bit => nsi [bit]
counter5b
  = do one <- vcc
       counter [one]
       
sim5b :: Sim [[Bool]]
sim5b = takeStream 10 counter5b
isim5b = stream2int sim5b

-------------------------------------------------------------------------------

delayChain :: Circuit nsi bit => 
              [bit] -> nsi [bit]
delayChain = delayBus 8 >-> delayBus 8 >-> delayBus 8

-------------------------------------------------------------------------------

nrList a b = transpose [i2bv 8 i | i <- [a..b]]
nrConst n k = [i2bv 8 k | i <- [1..]]



sim7 :: Sim [[Bool]]
sim7 = delayChain (nrList 4 9) 
isim7 = stream2int sim7

-------------------------------------------------------------------------------
     
-- This counter uses the Lava defined ripple carry adder.
-- Circuit: loop (ripAdd ; delay ; fork)

counter8 :: Circuit nsi bit => [bit] -> nsi [bit]
counter8 = loopList 8 counterCore8

counterCore8 :: Circuit nsi bit => 
                ([bit], [bit]) ->
                nsi ([bit], [bit])     
counterCore8 = snD (delayBus 8) >-> verticalAdder >-> fork2

-- Laziness test.

lt8 :: Sim [Bool]
lt8 = headStream (Sim f)
      where
      Sim (v,f) = counterCore8 ([replicate 8 True], undefined)

-------------------------------------------------------------------------------
       
sim8 :: Sim [[Bool]]
sim8 = takeStream 10 (counter8 ones)
isim8 = stream2int sim8
isym8 = showNetList (counter8 (vec "a" 8))

counter8Top
  = do a <- inputVec "a" (bit_vector 7 downto 0)
       clk <- inputBit "clk" BitType
       clr <- inputBit "clr" BitType
       r <- counter8 a
       outputVec r "r" (bit_vector 7 downto 0)

-------------------------------------------------------------------------------

counter8bCore a b = fork2 (delayBus 8 (ripAdd (a,b)))

counter8b :: Circuit nsi bit => [bit] -> nsi [bit]     
counter8b = loopList 8 (verticalAdder >-> delayBus 8 >-> fork2)

 
counter8bTop
  = do a <- inputVec "a" (bit_vector 7 downto 0)
       clk <- globalClock "clk"
       clr <- globalClear "clr"
       r <- counter8b a
       outputVec r "r" (bit_vector 7 downto 0)

sim8b :: Sim [[Bool]]
sim8b = takeStream 10 (counter8b ones)
isim8b = stream2int sim8b
isym8b = showNetList (counter8b (vec "a" 8))

-------------------------------------------------------------------------------

invchain4 :: Circuit nsi bit => bit -> nsi (bit)
invchain4 = inv >-> inv >-> inv >-> inv

invchain4top
  = do i <- inputBit "i" BitType
       o <- invchain4 i
       outputBit o "o"
       
invchain4nl = showNetList (invchain4 (Var "i"))       

-------------------------------------------------------------------------------

vchain4 :: Circuit nsi bit => bit -> nsi (bit)
vchain4 = inv /\ inv /\ inv -- /\ inv

       
vchain4nl = showNetList (vchain4 (Var "i"))       

-------------------------------------------------------------------------------

delay3 :: Circuit nsi bit => bit -> nsi bit
delay3 = delayBit False >-> delayBit False >-> delayBit False

delay3top
  = do a <- inputBit "a" BitType 
       c <- globalClock "c"
       d <- delay3 a
       outputBit d "d" 
       
d3c = circuit2VHDL "d3" delay3top       

-------------------------------------------------------------------------------
