-------------------------------------------------------------------------------
-- $Id: Simulation.hs,v 1.27 2000/01/20 18:24:32 satnam Exp $
-------------------------------------------------------------------------------

module Simulation
where

import Lava
import VirtexGates
import List
import IOExts 

-------------------------------------------------------------------------------

data Sim a = Sim a
             deriving Eq

-------------------------------------------------------------------------------

instance Show (Sim [Bool]) where
  showsPrec _ (Sim bits) = showString (showBV bits) 

-------------------------------------------------------------------------------

instance Show (Sim [[Bool]]) where
  showsPrec _ (Sim bits) = showString (showBVList bits) 

-------------------------------------------------------------------------------

instance (Show a, Show b) => Show (Sim (a, b)) where
  showsPrec _ (Sim (a, b)) 
    = showString ("("++ show a ++ ", " ++ show b ++ ")")	

-------------------------------------------------------------------------------


instance Monad Sim where
  (>>=) (Sim a) f
    = f a
  return a = Sim a     

-------------------------------------------------------------------------------

instance Stream [Bool] where
  bitvec2ints = bv2iS
  signed_bitvec2ints = sbv2iS
  to_bitvec = id

instance Spacers Sim

instance Combinational Sim [Bool] where
  gate1 f i                = Sim (map f i) 
  gate2 f (i0, i1)         = Sim (zipWith f i0 i1)
  gate3 f (i0, i1, i2)     = Sim (zipWith3 f i0 i1 i2) 
  gate4 f (i0, i1, i2, i3) 
    = Sim [f iv0 iv1 iv2 iv3 | (iv0, iv1, iv2, iv3) <- zip4 i0 i1 i2 i3]
  rom16x1 = rom16x1_sim
  gnd = return (repeat False)
  vcc = return (repeat True)
  buf i = return i
  xorcy = xor2
  muxcy (ss,(d0s,d1s)) = return [if s then d1 else d0 | (s,d0,d1)<- zip3 ss d0s d1s]
  delayBit initialValue stream 
    = Sim (initialValue:stream)
  delayBus = delayNBusSim
  fd = fdSim
  fdc = fdcSim  
  intConstant size v = Sim (repeat (i2bv size v)) 

 
-------------------------------------------------------------------------------

rom16x1_sim iv [a0,a1,a2,a3] 
  = Sim [iv!!(bv2i [x0,x1,x2,x3]) | (x0,x1,x2,x3) <- zip4 a0 a1 a2 a3]
rom16x1_sim _ addr
  = error ("rom16x1 SIM: address should be four bits: " ++ show addr ++ "\n")

-------------------------------------------------------------------------------

rom32x1_sim iv [a0,a1,a2,a3,a4] 
  = Sim [iv!!(bv2i [x0,x1,x2,x3,x4]) | (x0,x1,x2,x3,x4)
         <- zip5 a0 a1 a2 a3 a4]
rom32x1_sim _ addr
  = error ("rom32x1 SIM: address should be five bits: " ++ show addr ++ "\n")

-------------------------------------------------------------------------------
              
instance Combinators Sim
instance Circuit Sim [Bool]

-------------------------------------------------------------------------------

instance Loop Sim [Bool] where    
  loop = loopSim
  loopList = loopSimList    
  
-------------------------------------------------------------------------------

loopSim :: ((input,loopBack) -> Sim (output,loopBack)) -> 
             input -> Sim output
loopSim cir as
  = Sim cs
    where
    Sim ~(cs, bs) = cir (as, bs)

-------------------------------------------------------------------------------

loopSimList :: Int -> ((input,[loopBack]) -> Sim (output,[loopBack])) -> 
             input -> Sim output
loopSimList n cir as
  = Sim cs
    where
    Sim ~(cs, bs) = cir (as, [bs!!i | i <- [0..n-1]])
    
-------------------------------------------------------------------------------

type StreamLoopCir input output 
  = (Sim (input,[[Bool]]) -> Sim (output,[[Bool]])) 

-------------------------------------------------------------------------------

consStream :: [a] -> [[a]] -> [[a]]
consStream [] [] = []
consStream a@(x:xs) b@(s:ss)
  = (x:s) : (consStream xs ss)
  
-------------------------------------------------------------------------------
    
firstSlice :: [[a]] -> [[a]] 
firstSlice vs = [[head v] | v <- vs]    

restSlice vs = [tail v | v <- vs]

singleton xs = [[x] | x <- xs]

-------------------------------------------------------------------------------

addBits_List_BoolList :: ([[Bool]], [[Bool]]) -> [[Bool]]
addBits_List_BoolList (a,b) = a
                            
-------------------------------------------------------------------------------

runSim :: Sim a -> a
runSim cir
  = result
    where
    Sim result = cir

------------------------------------------------------------------------------- 

delayNBusSim n input 
  = Sim result
    where
    result = [False : (input!!i) | i <- [0..n-1]]

-------------------------------------------------------------------------------

stream2int x = map bv2i (transpose (runSim x)) 
stream2signedint x = map sbv2i (transpose (runSim x))

dispsigned x
  = putStr (unlines [reverse (showBV bv) ++ "#" ++ show (length bv) ++
            " : " ++ show (sbv2i bv) | bv <- bvs])
    where
    bvs = transpose (runSim x) 

dispunsigned x
  = putStr (unlines [reverse (showBV bv) ++ "# " ++ show (length bv) ++
            " : " ++ show (bv2i bv) | bv <- bvs])
    where
    bvs = transpose (runSim x)
     
-------------------------------------------------------------------------------

intStream size values = transpose [i2bv size i | i <- values]

-------------------------------------------------------------------------------

takeStream n (Sim stream) = Sim [checkTake "takeStream" n s | s <- stream]
headStream (Sim stream) = Sim [head s | s <- stream]

-------------------------------------------------------------------------------

fdSim :: [Bool] -> [Bool] -> Sim [Bool]
fdSim c d
  = Sim (False : fdSim' False (zip d c))
  
fdSim' q [] = []
fdSim' q ((d,c):rest)
  = if c == False then
       q : fdSim' q rest
     else
       d : fdSim' d rest  

-------------------------------------------------------------------------------

fdcSim :: [Bool] -> [Bool] -> [Bool] -> Sim [Bool]
fdcSim c clr d
  = Sim (False : fdcSim' False (zip3 d c clr))
  
fdcSim' q [] = []
fdcSim' q ((d,c,clr):rest)
  = if clr == True then
      False : fdcSim' False rest
    else 
      if c == False then
         q : fdcSim' q rest
       else
         d : fdcSim' d rest 

-------------------------------------------------------------------------------

clk = False : True : clk

-------------------------------------------------------------------------------
