-------------------------------------------------------------------------------
-- $Id: RippleAdder.hs,v 1.18 2000/01/15 01:02:31 satnam Exp $
-------------------------------------------------------------------------------

module RippleAdder
where
import Lava
import Virtex
import Registers

import Lava

-------------------------------------------------------------------------------

type TwosComplement bit = [bit]

-------------------------------------------------------------------------------

rippleAdder' :: Circuit nsi bit =>
                [bit] -> [bit] -> bit -> nsi [bit]
rippleAdder' [] [] cin = return [cin]
rippleAdder' (a:as) [] cin
  = do sum <- xorcy (a, cin)
       cout <- muxcy (a, (a, cin))
       rest <- rippleAdder' as [] cout
       return (sum : rest)
rippleAdder' [] (b:bs) cin
  = do zero <- gnd
       sum <- xorcy (b, cin)
       cout <- muxcy (b, (zero, cin))
       rest <- rippleAdder' [] bs cout
       return (sum : rest) 
rippleAdder' (a:as) (b:bs) cin
  = do part_sum <- xor2 (a, b)
       sum <- xorcy (part_sum, cin)
       cout <- muxcy (part_sum, (a, cin))
       rest <- rippleAdder' as bs cout
       return (sum : rest)

-------------------------------------------------------------------------------

equaliseVectors x y
  = if length x == length y then
       return (x, y)
     else
       if length x > length y then
        equaliseVectors y x
      else
        do zero <- gnd
           return (x++replicate padding zero, y)
    where
    padding = length y - length x


-------------------------------------------------------------------------------

equaliseSignedVectors x y
  = if length x == length y then
       return (x, y)
     else
       if length x > length y then
        equaliseSignedVectors y x
      else
        do let pad = last x
           return (x++replicate padding pad, y)
    where
    padding = length y - length x

-------------------------------------------------------------------------------
  
flexibleAdder :: Circuit nsi bit =>
                 (TwosComplement bit, TwosComplement bit) -> 
                  nsi (TwosComplement bit)
flexibleAdder (x, y)
  = do (xe, ye) <- equaliseVectors x y
       ripAdd (xe, ye)
    where
    size = max (length x) (length y) 


-------------------------------------------------------------------------------

flexibleSignedAdder :: Circuit nsi bit =>
                       (TwosComplement bit, TwosComplement bit) -> 
                       nsi (TwosComplement bit)
flexibleSignedAdder (x, y)
  = do (xe, ye) <- equaliseSignedVectors x y
       ripAdd (xe, ye)
    where
    size = max (length x) (length y) 

-------------------------------------------------------------------------------

registeredAdder :: Circuit nsi bit =>
                   (TwosComplement bit, TwosComplement bit) -> 
                    nsi (TwosComplement bit)
registeredAdder = flexibleAdder >-> register                    

-------------------------------------------------------------------------------

as, bs :: [[Bool]]
as = ints2stream 8 [0, 1,  2, 3, 4, 5, 6]
bs = ints2stream 8 [0, 12,13,14,15,16,17]
cins = replicate (length as) False

ints2stream size ints = transpose [i2bv size i | i <- ints] 

rt1 :: [Int]
rt1 = stream2int (rippleAdder' as bs cins)

errors :: [(Int,Int)]
errors
  = diffs expected r
    where
    vecs = [((a,b),a+b) | a <- [0..nr-1], b <- [0..nr-1]]
    at = ints2stream size (map (fst . fst) vecs)
    bt = ints2stream size (map (snd . fst) vecs)
    r2 = stream2int (rippleAdder' at bt zeros)
    r = stream2int (ripAdd (at, bt))
    expected = map snd vecs
    size = 4
    nr = 2^size
    zeros = repeat False
     
diffs :: [Int] -> [Int] -> [(Int,Int)]     
diffs [] [] = []
diffs (a:as) (b:bs)
  = if a == b then
     diffs as bs
    else
      (a,b) : diffs as bs
      

rasym1 = runSymbolic (rippleAdder' sym_avec4 sym_bvec4 (Variable "cin"))

sym_vec a n = [Variable (a++show i) | i <- [0..n-1]]

sym_avec4 = sym_vec "a" 4 
sym_bvec4 = sym_vec "b" 4 

ripAdd :: Circuit nsi bit =>
          (TwosComplement bit, TwosComplement bit) ->
          nsi (TwosComplement bit)
ripAdd ab 
  = do zero <- gnd
       r <- rippleAdder' a b zero
       return r
    where
    (a,b) = ab

rippAddDelayed :: Circuit nsi bit =>
                  (TwosComplement bit, TwosComplement bit) ->
                  nsi (TwosComplement bit)     
rippAddDelayed = snD (delayBus 8) >-> ripAdd  

lt8b :: Sim [Bool]
lt8b = headStream f
       where
       f = rippAddDelayed ([replicate 8 True], undefined) 
       
       
lt8c :: Sim [Bool]
lt8c = headStream f
       where
       f = delayBus 8 undefined 
       
       
rippAddDelayed2 :: [[Bool]] -> [[Bool]] -> Sim [[Bool]]     
rippAddDelayed2 a b
   = rippleAdder' a b (repeat False) 
   
lt8d :: Sim [Bool]
lt8d = headStream f
       where
       f = rippAddDelayed2 [replicate 8 True] undefined
       
       
-------------------------------------------------------------------------------

-- Tests for a 1-bit ripple-adder

-- First create a netlist description of a 1-bit adder.

add1bit :: StateST s [SignalA String]
add1bit = ripAdd ([Var "a"], [Var "b"])

-- Now generate the netlist.

add1nl = showNetList add1bit

-------------------------------------------------------------------------------

-- A full-adder tile suitable for use by the COL combinator.

fullAdder :: Circuit nsi bit =>
             (bit, (bit, bit)) -> nsi (bit, bit)
fullAdder (cin, (a,b))
  = do part_sum <- xor2 (a, b)
       sum <- xorcy (part_sum, cin)
       cout <- muxcy (part_sum, (a, cin))
       return (sum, cout)
       
-------------------------------------------------------------------------------

verticalAdder (x, y)
  = do (xe, ye) <- equaliseVectors x y
       zero <- gnd
       (sum, cout) <- col size fullAdder (zero, zip xe ye)
       return (sum ++ [cout])
    where
    size = max (length x) (length y) 


-------------------------------------------------------------------------------

verticalAdderNoGrowth (x, y)
  = do (xe, ye) <- equaliseVectors x y
       zero <- gnd
       (sum, cout) <- col size fullAdder (zero, zip xe ye)
       return sum
    where
    size = max (length x) (length y) 

-------------------------------------------------------------------------------

verticalSignedAdder (x, y)
  = do (xe, ye) <- equaliseSignedVectors x y
       zero <- gnd
       (sum, cout) <- col size fullAdder (zero, zip xe ye)
       return (sum ++ [cout])
    where
    size = max (length x) (length y) 

-------------------------------------------------------------------------------


-- Tests for a 4-bit ripple-adder



add4 :: StateST s [SignalA String]
add4 = verticalAdder (vec "a" 4, vec "b" 4)

-- Now generate the netlist.

add4nl = showNetList add4Top

add4Top :: StateST state ()
add4Top
  = do a <- inputVec "a" (bit_vector 3 downto 0)
       b <- inputVec "b" (bit_vector 3 downto 0)
       r <- verticalAdder (a, b)
       outputVec r "c" (bit_vector 3 downto 0)
       
add4v = circuit2VHDL "add4" add4Top       

rt2 :: [Int]
rt2 = stream2int (verticalAdder (as, bs))
-- Expected output: [0,13,15,17,19,21,23]

-------------------------------------------------------------------------------

complement :: Circuit nsi bit => TwosComplement bit -> nsi (TwosComplement bit)
complement = invert >-> add1

-------------------------------------------------------------------------------

invert :: Circuit nsi bit => [bit] -> nsi [bit]
invert = maP inv

-------------------------------------------------------------------------------

add1 :: Circuit nsi bit => TwosComplement bit -> nsi (TwosComplement bit)
add1 a
   = do one <- vcc
        let b = [one]
        r <- verticalAdderNoGrowth (a, b)
        return r
 

-------------------------------------------------------------------------------

mixedSignAdder :: Circuit nsi bit => 
                 ((TwosComplement bit, Bool), (TwosComplement bit, Bool)) ->
                 nsi (TwosComplement bit, Bool)
mixedSignAdder ((a, aSigned), (b, bSigned))
  = do aExt <- extendNumber extendTo a aSigned
       bExt <- extendNumber extendTo b bSigned
       r <- verticalAdderNoGrowth (aExt, bExt)
       return (r, aSigned || bSigned)
    where
    extendTo = max (length a) (length b)

-------------------------------------------------------------------------------

extendNumber :: Circuit nsi bit =>
                Int -> TwosComplement bit -> Bool -> nsi (TwosComplement bit)
extendNumber extendTo a False
  = do zero <- gnd
       return (a ++ replicate (extendTo - length a) zero)
       
extendNumber extendTo a True
  = return (a ++ replicate (extendTo - length a) topBit)
    where
    topBit = last a
    
-------------------------------------------------------------------------------
                                    