-------------------------------------------------------------------------------
-- $Id: SplitMultiplier.hs,v 1.19 2000/01/21 17:51:17 satnam Exp $
-------------------------------------------------------------------------------

module MotorolaKCM (motkcm_top, motkcm, mot_latency)
where
import Lava
import Virtex
import VirtexLib
import IOExts
import LUTROM
import KCMUtils
import SplitMultiplier

-------------------------------------------------------------------------------

-- A Split-Multiplier KCM [Motorola]

motkcm_top :: Integral num => Bool -> num -> Int -> Int ->
              StateST state ()
motkcm_top dataSigned k kSize  dataSize
  = do clk <- globalClock "c"
       d <- inputVec "a" (bit_vector (dataSize-1) downto 0)
       p <- motkcm dataSigned k kSize d
       outputVec p "prod" (bit_vector (dataSize+kSize-1) downto 0)  

-------------------------------------------------------------------------------

mot_latency :: Int -> Int
mot_latency dataSize
  = case dataSize of
                5  -> 1
                10 -> 2
                other  -> latency other 
                      
-------------------------------------------------------------------------------

fiveBitMultiplier :: (Integral num, Circuit nsi bit) =>
                     num -> Int -> [bit] -> nsi ([bit], Bool)
fiveBitMultiplier coief coief_width addr
  = do zero <- gnd
       let padded_addr = addr ++ (replicate padding zero)
       r <- rom32x (coief_width+nr_addrs)
                    multiplication_results padded_addr
       return (notrace ("multrel = " ++ show multiplication_results)
              (r, signed_result))  
    where
    multiplication_results 
      = pad_with 0 32 [coief * i | i <- [0..2^nr_addrs-1]]
    nr_addrs = length addr
    padding = 5-nr_addrs
    signed_result = coief < 0

-------------------------------------------------------------------------------

signedFiveBitMultiplier coief coief_width addr
  = do zero <- gnd
       let padded_addr = addr ++ (replicate padding zero)
       r <- rom32x (coief_width+nr_addrs) multiplication_results padded_addr
       return (r, True)  
    where
    multiplication_results 
      = pad_with 0 32 [coief * aAddr | aAddr <- aAddrs]
    aAddrs = [sbv2i (i2bv nr_addrs i) | i <- [0..2^nr_addrs-1]]                
    nr_addrs = length addr
    padding = 5-nr_addrs

-------------------------------------------------------------------------------

motkcm :: (Integral num, Circuit nsi bit) =>
                   Bool -> num -> Int -> [bit] -> 
                   nsi [bit]
                    
-- Five-bit case
motkcm signed_data coief coief_width addr | length addr == 5 
  = splitBy5Multiplier signed_data coief coief_width addr 
-- Ten-bit case
motkcm signed_data coief coief_width addr | length addr == 10 
  = splitBy5Multiplier signed_data coief coief_width addr
-- Seventeen-bit case
motkcm signed_data coief coief_width addr | length addr == 17
  = splitBy5Multiplier signed_data coief coief_width addr
motkcm signed_data coief coief_width addr
  = splitKCM signed_data coief coief_width addr 

-------------------------------------------------------------------------------

-- A pipelined split-multiplier KCM that uses 5-bit look-up tables. 
splitBy5Multiplier :: (Integral num, Circuit nsi bit) =>
                   Bool -> num -> Int -> [bit] -> 
                   nsi [bit]              
splitBy5Multiplier signed_data coief coief_width addr 
  = do let chopped_addr = chop 5 addr
       let chops = length chopped_addr 
       -- Build the lookup tables for the partial products
       let last_lut = if signed_data then
                          signedFiveBitMultiplier
                      else
                          fiveBitMultiplier
       let luts = replicate (chops-1) fiveBitMultiplier ++ [last_lut]  
       let regluts = hpar [mult5 coief coief_width >|> fsT (pitchedRegister 2) |
                           mult5 <- luts] 
       (w, s, product) <- (regluts >-> 
                           insertWeights >-> 
                           pipeTree reg3 weightedAdder) 
                           chopped_addr
       return product
    where
    addr_size = length addr
    -- The weights are a sequence [0,5,10,...] which indicate how much
    -- a group of bits has been shifted by 
    weights = [5*i | i <- [0..]]
    insertWeights pp
      = notrace ("pp:\n" ++ showPPs pp) 
        (return [(w,s,p) | ((p,s),w) <- zip pp weights])
    signed_coief = coief < 0
    signed_result = signed_coief || signed_data


-------------------------------------------------------------------------------

