-------------------------------------------------------------------------------
-- $Id: EDIF.hs,v 1.12 2000/01/23 03:30:15 satnam Exp $
-------------------------------------------------------------------------------

module EDIF (circuit2EDIF)
where
import Lava
import VirtexNetlist
import Array
import HexUtils
import RPM
import IOExts

-------------------------------------------------------------------------------

circuit2EDIF :: String -> NetList -> IO ()
circuit2EDIF name tuples@(instCount, netCount, intf', 
                          (clk,clr,zero,one), instances) 
  = do putStr ("Generating EDIF    " ++ name ++ ".edif...")
       writeFile filename (unlines (edif_header name))
       appendFile filename (unlines libraryCells)
       appendFile filename (unlines (edif_post_lib))
       appendFile filename (unlines (cell_edif name intf))
       appendFile filename (unlines (edif_instances instances netCount))
       edif_nets filename drivers tuples
       edif_nets filename outputs tuples
       appendFile filename (unlines (edif_trailer name)) 
       putStrLn ""
    where
    filename = name ++ ".edif"
    intf = reverse intf'
    externalCells = findLibraryCells [] tuples 0
    libraryCells = concat (map library_cell externalCells)
    inputs  = smashInputs intf
    outputs = smashOutputs intf 
    drivers = notrace ("outputs = " ++ show outputs) 
              (inputs ++
              [LavaSig i | i <- [0..netCount-1]])
                            
-------------------------------------------------------------------------------

smashInputs :: [Port] -> [BitSignal]
smashInputs intf 
  = concat [smash (name, typ) | i@(Port Input name typ) <- intf]

smashOutputs :: [Port] -> [BitSignal]
smashOutputs intf 
  = concat [smash (name, typ) | o@(Port Output name typ) <- intf]

-------------------------------------------------------------------------------

smash (name, BitType) = [Var name]
smash (name, typ@(BitVector a dir b))
   = [ArrayElement typ (Var name) i | i <- [min a b .. max a b]]
     where
     idx a To b = [a..b]
     idx a Downto b = reverse [b..a]

-------------------------------------------------------------------------------

edif_header name
  = ["(edif " ++ name,
     " (edifVersion 2 0 0)",
     " (edifLevel 0)",
     " (keywordMap (keywordLevel 0))",
     " (status",
     "  (written (timeStamp 2000 11 19 15 39 43)",
     "   (program \"Lava\" (Version \"2000.14\"))",
     "   (dataOrigin \"Xilinx-Lava\") (author \"Xilinx Inc.\")",
     "  )",
     " )",
     " (external lava_virtex_lib",
     "  (edifLevel 0)",
     "  (technology (numberDefinition))"]
          
-------------------------------------------------------------------------------

edif_post_lib
  = [" )",
     " (library DESIGNS (edifLevel 0) (technology (numberDefinition))"]  

-------------------------------------------------------------------------------

library_cell (name, intf)
  = ["  (cell " ++ name ++ " (cellType GENERIC)",
     "   (view " ++ prim ++ " (viewType NETLIST)"] ++
     interface_to_edif intf ++
    ["    )",
     "   )",
     "  )"]
     
-------------------------------------------------------------------------------

prim = "prim"

-------------------------------------------------------------------------------

cell_edif name intf
  = [" (cell " ++ name ++ " (cellType GENERIC)",
     "   (view " ++ prim ++ " (viewType NETLIST)"] ++
     interface_to_edif intf ++
     ["    )",
      "    (contents"]   

-------------------------------------------------------------------------------

interface_to_edif :: [Port] -> [String]
interface_to_edif intf
  = ["    (interface "] ++
    (indent 6 . map show_interface_edif) intf

-------------------------------------------------------------------------------

show_interface_edif :: Port -> String
show_interface_edif (Port Input name typ)
   = "     (port " ++ show_port_name name typ ++ " (direction " ++
      "INPUT" ++ "))"
show_interface_edif (Port Output name typ)
   = "     (port " ++ show_port_name name typ ++ " (direction " ++
      "OUTPUT" ++ "))"
      
show_port_name name BitType = name
show_port_name name (BitVector a _ b)
  = "(array (rename " ++ name ++ " \"" ++ name ++ "(" ++ show a ++ ":" ++  
    show b ++ ")\") " ++ show (abs (a-b) +1) ++ ")"
    
    
    
-------------------------------------------------------------------------------

type CellInterface = (String, [Port])

findLibraryCells :: [CellInterface] -> NetList -> Int -> [CellInterface]
findLibraryCells cells (_, n, _, _, _) i | i == n = cells
findLibraryCells alreadyFound nl@(_, n, _, _, netlist) i
  = if cintf `elem` alreadyFound || cell == Unused then
      findLibraryCells alreadyFound nl (i+1)
    else
      findLibraryCells (cintf:alreadyFound) nl (i+1)  
    where
    (cell, _, _, intf) = netlist!i
    cintf = (showCell cell, portmap2port intf)

-------------------------------------------------------------------------------
 
edif_instances :: Array Int InstanceTuple -> Int -> [String] 
edif_instances instances n 
  = concat [edif_instance (instances!i) i | i <- [0..n-1]]

-------------------------------------------------------------------------------

edif_instance :: InstanceTuple -> Int -> [String]
edif_instance (Unused, pos, attrs, portmaps) instNr = []
edif_instance (Buf, pos, attrs, portmaps) instNr | pos /= Unplaced
  = edif_instance (Buf, Unplaced, attrs, portmaps) instNr
edif_instance (cell, pos, attrs, portmaps) instNr
  = ["       (instance " ++ instName,
     "         (viewRef " ++ prim,
     "          (cellRef " ++ cellName ++
                 " (libraryRef lava_virtex_lib))",
     "         )"] ++
    
     process_attributes attrs ++
     process_rloc pos ++
     [
     "       )"]
    where
    cellName = showCell cell
    instName = cellName ++ "_" ++ show instNr
edif_instance _ _ = []    


-------------------------------------------------------------------------------

process_attributes :: [Property] -> [String]
process_attributes = concat . map process_attr

-------------------------------------------------------------------------------

process_attr (InitialBV size initv)
  = ["           (property INIT (string \"" ++ initStr' ++ "\"))"]
    where
    initStr = dropWhile ((==)'0') (wrHex 10 initv) 
    initStr' = if initStr == "" then
                 "0"
               else
                 initStr
process_attr _ = []
  
-------------------------------------------------------------------------------

process_rloc Unplaced = []
process_rloc (Position p)
  = ["        (property RLOC (string \"" ++ showRLOC p ++ "\"))"]
process_rloc (USet set p o)
  = ["        (property RLOC (string \"" ++ showRLOC p ++ "\"))",
     "        (property HU_SET (string \"" ++ set ++ "\"))"]

-------------------------------------------------------------------------------

edif_nets :: String -> [BitSignal] -> NetList ->IO ()
edif_nets filename drivers (_, n, _, _, netlist)
  = sequence_ [appendFile filename (unlines (edif_net d netlist n)) 
               | d <- drivers]  

-------------------------------------------------------------------------------

edif_net :: BitSignal -> Array Int InstanceTuple -> Int -> [String]
edif_net driver instances n
  = if joins == [] then
      notrace ("edif_net: " ++ show driver ++ " has no joins.\n") []
    else
      ["        (net " ++ show_edif_net_name driver,
       "         (joined"] ++
      show_edif_port  driver ++
      joins ++
      [
       "         )",
       "        )"
      ]
    where
    joins = concat [find_join driver (instances!i) i | i <- [0..n-1]]

-------------------------------------------------------------------------------

find_join :: BitSignal -> InstanceTuple -> Int -> [String]
find_join driver (cell, pos,  _, intf) inst_nr
  = ["        (portRef " ++ par ++ " (instanceRef " ++ instName ++ "))" |
     PortMap _ par arg <- intf, driver == arg]
    where
    cellName = showCell cell
    instName = cellName ++ "_" ++ show inst_nr
 
    
-------------------------------------------------------------------------------

show_edif_port v
  = if isLavaSig v then
      []
    else
       ["        (portRef " ++ show_edif_var v ++ ")"] 
       
-------------------------------------------------------------------------------

isLavaSig (LavaSig _) = True
isLavaSig other = False
    
-------------------------------------------------------------------------------

show_edif_net_name (ArrayElement typ (Var v) i)
  = "(rename " ++ renamed ++ " \"" ++ v ++  "(" ++ show i ++ ")\")"
    where
    renamed = v ++ "_" ++ show i
show_edif_net_name other = show_edif_var other  

show_edif_var (Var v) = v
show_edif_var (ArrayElement typ (Var v) i) 
  =  showEdifArrayElement typ v i 
show_edif_var (LavaSig i) = "lava_bit" ++ show i

-------------------------------------------------------------------------------

showEdifArrayElement (BitVector a To b) name i
  = "(member " ++ name ++ show (i-a) ++ ")"
showEdifArrayElement (BitVector a Downto b) name i  
  = "(member " ++ name ++  " " ++ show (a-i) ++ ")"
  
-------------------------------------------------------------------------------

isVar (Var _) = True
isVar other = False  

-------------------------------------------------------------------------------

edif_trailer name
  = [
     "    )",
     "   )",
     "  )",
     " )",
     " (design " ++ name ++ " (cellRef " ++ name ++ " (libraryRef DESIGNS)))",
     ")"
    ]

-------------------------------------------------------------------------------
                                            
