module BitOps (i2bv, bv2i, sbv2i, showBV, showBVList, BITClass,
               int2bitvec, bitvec2Num, i2bvS, bv2iS, sbv2iS)
where
import List
import Utils

--------------------------------------------------------------------------------
-- $Id: BitOps.hs,v 1.6 2000/01/20 18:24:31 satnam Exp $
--------------------------------------------------------------------------------

class (Eq a, Integral a) => BITClass a where
  l, h :: a
  fromBIT :: a -> Int

--------------------------------------------------------------------------------

data BIT = L | H
           deriving Eq

--------------------------------------------------------------------------------

instance Show BIT where
  showsPrec n v = showChar (showBIT v)

--------------------------------------------------------------------------------

instance BITClass BIT where
  l = L
  h = H
  fromBIT L = 0
  fromBIT H = 1

-------------------------------------------------------------------------------

instance Num BIT where
  (+) = logor
  (*) = logand
  negate = lognot
  fromInteger 0 = L
  fromInteger 1 = H

  
instance Enum BIT where
  toEnum 0 = L
  toEnum 1 = H
  fromEnum L = 0
  fromEnum H = 1

instance Ord BIT where
  compare L H = LT
  compare H L = GT
  compare L L = EQ
  compare H H = EQ
    
  
instance Real BIT where
  toRational L = 0
  toRational H = 1   

instance Integral BIT where
  toInteger L = 0
  toInteger H = 1
  
-------------------------------------------------------------------------------

instance Num Bool where
  fromInteger 0 = False
  fromInteger 1 = True

-------------------------------------------------------------------------------

instance Real Bool where
  toRational False = 0
  toRational True =  1

-------------------------------------------------------------------------------


instance Integral Bool where
  toInteger False = 0
  toInteger True  = 1

-------------------------------------------------------------------------------


instance BITClass Bool where
  l = False
  h = True
  fromBIT False = 0
  fromBIT True = 1

-------------------------------------------------------------------------------

instance BITClass Int where
 l = 0
 h = 1
 fromBIT 0 = 0
 fromBIT 1 = 1

--------------------------------------------------------------------------------

instance BITClass Integer where
 l = 0
 h = 1
 fromBIT 0 = 0
 fromBIT 1 = 1

-------------------------------------------------------------------------------

type BITVec a = [a]

--------------------------------------------------------------------------------

showBIT :: BITClass a => a -> Char
showBIT bit = if bit == h then '1' else '0'

--------------------------------------------------------------------------------

showBITVec :: BITClass bit => BITVec bit -> String
showBITVec = reverse . map showBIT

--------------------------------------------------------------------------------

-- Conversion fuctions between bits and booleans.

bitToBool :: BITClass a => a -> Bool
bitToBool v
  = if v == l then
      False
    else
      True

boolToBIT :: BITClass a => Bool -> a
boolToBIT False = l
boolToBIT True = h

--------------------------------------------------------------------------------

-- This function takes an unsigned bit-vector and returns the 
-- corresponding unsigned integer. Note that the LSB is the first
-- element of the list.

bitvec2Num :: (Num n, BITClass a) => BITVec a -> n
bitvec2Num bits = sum [n | (n, b) <- zip powersOf2 bits, b == h]

-------------------------------------------------------------------------------

-- Signed bit-vector to integer (general)

signedbitvec2int :: (BITClass b, Num num) => [b] -> num
signedbitvec2int bits
  = sum [n | (n, b) <- z, b == h]
    where
    z = zip powersOf2 (init bits) ++ [(-2^(length bits-1), last bits)]

-------------------------------------------------------------------------------

-- Signed bit-vector to integer.
-- Example: sbv2i [1,1,1,1] = -1

sbv2i :: Num num => [Bool] -> num
sbv2i bits
  = sum [n | (n, b) <- z, b == h]
    where
    z = zip powersOf2 (init bits) ++ [(-2^(length bits-1), last bits)]

-------------------------------------------------------------------------------


bv2i :: Num num => [Bool] -> num
bv2i = bitvec2Num

bv2int :: [Int] -> Int
bv2int = bitvec2Num

-------------------------------------------------------------------------------

bv2n :: (BITClass a, Num b) => BITVec a -> b
bv2n = fromInteger . bitvec2Num

--------------------------------------------------------------------------------

powersOf2 :: Num n => [n]
powersOf2 = powersOf2' 1
powersOf2' n = n : powersOf2' (2*n)

--------------------------------------------------------------------------------

-- Conversion functions between bits and integers

bit2Integer :: (BITClass a, Num n) => a -> n
bit2Integer b = if b == l then 0 else 1

int2BIT :: (BITClass a, Num n) => n -> a
int2BIT 0 = l
int2BIT 1 = h

--------------------------------------------------------------------------------

-- This function takes an integer and a bit-vector width and returns the
-- corresponding bit-vetctor. Only works for unsigned values, and the
-- first element of the list is the LSB.

int2bitvec :: (Integral n, BITClass b) => Int -> n -> BITVec b
int2bitvec 0 0 = []
int2bitvec 0 _ = error ("int2bitvec: bit-vector too small\n")
int2bitvec n v
  = if v < 0 then
        signed_int2bit n v
    else 
      if n < bitsrequired v then
        error ("int2bitvec " ++ show n ++ " " ++ show v ++ " : " ++
               "need at least " ++ show (bitsrequired v) ++ " bits.")
      else 
       int2BIT (v `mod` 2) : (int2bitvec (n-1) (v `div` 2))

-- ? (int2bitvec 8 25)::[BIT]
-- [1, 0, 0, 1, 1, 0, 0, 0]
 

-------------------------------------------------------------------------------

signed_int2bit :: (Integral n, BITClass b) => Int -> n -> BITVec b
signed_int2bit n v
  = int2bitvec n twos_comp
    where
    bits = (int2bitvec n (abs v))::[Bool]
    bits' = map lognot bits
    comp = bitvec2Num bits'
    twos_comp = (comp + 1) `mod` (2^n)
    
si2bv :: Integral nr => Int -> nr -> [Bool] 
si2bv = signed_int2bit 

--------------------------------------------------------------------------------

i2bv :: Integral nr => Int -> nr -> [Bool]
i2bv = int2bitvec

--------------------------------------------------------------------------------

lognot :: BITClass a => a -> a
lognot b = if b == l then h else l

--------------------------------------------------------------------------------

logand :: BITClass a => a -> a -> a
logand a b = if a == h && b == h then h else l

--------------------------------------------------------------------------------

logor :: BITClass a => a -> a -> a
logor a b = if a == l && b == l then l else h

--------------------------------------------------------------------------------

logexor :: BITClass bit => bit -> bit -> bit
logexor a b = if a == b then l else h

--------------------------------------------------------------------------------
showBoolBit False = '0'
showBoolBit True = '1'

showBV = map showBoolBit

--------------------------------------------------------------------------------
showBVList = show . map showBV 

--------------------------------------------------------------------------------
-- Operations over streams

i2bvS size values = transpose [int2bitvec size i | i <- values]

bv2iS x = map bitvec2Num (transpose x)
sbv2iS x = map signedbitvec2int (transpose x)

--------------------------------------------------------------------------------     
