/* rtl8019.c: A general non-shared-memory RTL8019 ethernet driver for linux. */
/*
    Written 1992-94 by Donald Becker. 

    Copyright 1993 United States Government as represented by the
    Director, National Security Agency.

    This software may be used and distributed according to the terms
    of the GNU Public License, incorporated herein by reference.

*/

/* Routines for the Realtek-based designs (RTL8019). */

static char *version =
    "rtl8019.c:v1.00 10/23/96 \n";

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/pci.h>
#include <linux/bios32.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/config.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <asm/bitops.h>
#include <asm/dma.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include "8390.h"


extern struct device *init_etherdev(struct device *dev, int sizeof_private,
				    unsigned long *mem_startp);



#define NE_BASE	 (dev->base_addr)
#define NE_CMD	 	0x00
#define NE_DATAPORT	0x10	/* NatSemi-defined port window offset. */
#define NE_RESET	0x1f	/* Issue a read to reset, a write to clear. */
#define NE_IO_EXTENT	0x20

#define NE1SM_START_PG	0x20	/* First page of TX buffer */
#define NE1SM_STOP_PG 	0x40	/* Last page +1 of RX ring */
#define NESM_START_PG	0x40	/* First page of TX buffer */
#define NESM_STOP_PG	0x80	/* Last page +1 of RX ring */

#define NE_RDC_TIMEOUT	0x02	/* Max wait in jiffies for Tx RDC */
static unsigned int irq_line;

int r8e_probe(struct device *dev);
static int r8e_probe1(struct device *dev, int ioaddr);

static void r8e_reset_8390(struct device *dev);
static int r8e_block_input(struct device *dev, int count,
			  char *buf, int ring_offset);
static void r8e_block_output(struct device *dev, const int count,
		const unsigned char *buf, const int start_page);



#ifdef HAVE_DEVLIST
struct netdev_entry netcard_drv =
{"r8e", r8e_probe1, NE_IO_EXTENT};
#else

int r8e_probe(struct device *dev)
{
    unsigned int tmp_irq,exist=0;
    unsigned int pnp_addr=0x200;
    while (pnp_addr <0x400)
      {
      /* Map the 'P' and 'p' in the configuration table */
      if((inb(pnp_addr+10)=='P') && (inb(pnp_addr+11)=='p'))
        {
        outw_p(0xe1,pnp_addr);
        tmp_irq=inb_p(pnp_addr+4);    /* fetch the IRQ value */
        tmp_irq=((tmp_irq & 0x70) >> 4);
        if(tmp_irq==0)
          irq_line=9;
        else if (tmp_irq <4)
          irq_line=tmp_irq+2;
        else if (tmp_irq==7)
          irq_line=15;
        else
          irq_line=tmp_irq+6; 
        outw_p(0x21,pnp_addr);
        r8e_probe1(dev, pnp_addr);
        exist=1;
        }
     pnp_addr=pnp_addr+0x20;         /* Increase IO base for the next search  */
     }
     if(exist==1)                    /* 8019 adapter exist */
       return 0;
    
    return ENODEV;
}
#endif

static int r8e_probe1(struct device *dev, int ioaddr)
{
    int i;
    unsigned char SA_prom[32];
    int wordlength = 2;
    char *name = NULL;
    int start_page, stop_page;
    int reg0 = inb_p(ioaddr);

    if (reg0 == 0xFF)
	return ENODEV;

    /* Do a preliminary verification that we have a 8390. */
    {	int regd;
	outb_p(E8390_NODMA+E8390_PAGE1+E8390_STOP, ioaddr + E8390_CMD);
	regd = inb_p(ioaddr + 0x0d);
	outb_p(0xff, ioaddr + 0x0d);
	outb_p(E8390_NODMA+E8390_PAGE0, ioaddr + E8390_CMD);
	inb_p(ioaddr + EN0_COUNTER0); /* Clear the counter by reading. */
	if (inb_p(ioaddr + EN0_COUNTER0) != 0) {
	    outb_p(reg0, ioaddr);
	    outb_p(regd, ioaddr + 0x0d);	/* Restore the old values. */
	    return ENODEV;
	}
    }

    printk("RTL8019 ethercard probe at %#3x:", ioaddr);

    /* Read the 16 bytes of station address PROM.
       We must first initialize registers, similar to NS8390_init(eifdev, 0).
       We can't reliably read the SAPROM address without this.
       (I learned the hard way!). */
    {
	struct {unsigned char value, offset; } program_seq[] = {
	    {E8390_NODMA+E8390_PAGE0+E8390_STOP, E8390_CMD}, /* Select page 0*/
	    {0x48,	EN0_DCFG},	/* Set byte-wide (0x48) access. */
	    {0x00,	EN0_RCNTLO},	/* Clear the count regs. */
	    {0x00,	EN0_RCNTHI},
	    {0x00,	EN0_IMR},	/* Mask completion irq. */
	    {0xFF,	EN0_ISR},
	    {E8390_RXOFF, EN0_RXCR},	/* 0x20  Set to monitor */
	    {E8390_TXOFF, EN0_TXCR},	/* 0x02  and loopback mode. */
	    {32,	EN0_RCNTLO},
	    {0x00,	EN0_RCNTHI},
	    {0x00,	EN0_RSARLO},	/* DMA starting at 0x0000. */
	    {0x00,	EN0_RSARHI},
	    {E8390_RREAD+E8390_START, E8390_CMD},
	};
	for (i = 0; i < sizeof(program_seq)/sizeof(program_seq[0]); i++)
	    outb_p(program_seq[i].value, ioaddr + program_seq[i].offset);

    }
    for(i = 0; i < 32 /*sizeof(SA_prom)*/; i+=2) {
	SA_prom[i] = inb(ioaddr + NE_DATAPORT);
	SA_prom[i+1] = inb(ioaddr + NE_DATAPORT);
    }

	/* We must set the 8390 for word mode. */
	outb_p(0x49, ioaddr + EN0_DCFG);
	/* We used to reset the ethercard here, but it doesn't seem
	   to be necessary. */
	/* Un-double the SA_prom values. */
	for (i = 0; i < 16; i++)
	    SA_prom[i] = SA_prom[i+i];
	start_page = NESM_START_PG;
	stop_page = NESM_STOP_PG;

    for(i = 0; i < ETHER_ADDR_LEN; i++) {
	dev->dev_addr[i] = SA_prom[i];
	printk(" %2.2x", SA_prom[i]);
    }


    /* Set up the rest of the parameters. */
    name = "RTL8019";

    dev = init_etherdev(0, sizeof(struct ei_device), 0);

    dev->irq=irq_line;  
    
    {
	int irqval = request_irq (dev->irq, ei_interrupt, 0, "RTL8019");
	if (irqval) {
	    printk (" unable to get IRQ %d (irqval=%d).\n", dev->irq, irqval);
	    return EAGAIN;
	}
    }
    dev->base_addr = ioaddr;

    request_region(ioaddr, NE_IO_EXTENT, "RTL8019");

    for(i = 0; i < ETHER_ADDR_LEN; i++)
	dev->dev_addr[i] = SA_prom[i];

    ethdev_init(dev);
    printk("\n%s: %s found at %#x, using IRQ %d.\n",
	   dev->name, name, ioaddr, dev->irq);

    if (ei_debug > 0)
	printk(version);

    ei_status.name = name;
    ei_status.tx_start_page = start_page;
    ei_status.stop_page = stop_page;
    ei_status.word16 = (wordlength == 2);

    ei_status.rx_start_page = start_page + TX_PAGES;
#ifdef PACKETBUF_MEMSIZE
    /* Allow the packet buffer size to be overridden by know-it-alls. */
    ei_status.stop_page = ei_status.tx_start_page + PACKETBUF_MEMSIZE;
#endif

    ei_status.reset_8390 = &r8e_reset_8390;
    ei_status.block_input = &r8e_block_input;
    ei_status.block_output = &r8e_block_output;
    NS8390_init(dev, 0);
    return 0;
}

/* Hard reset the card.  This used to pause for the same period that a
   8390 reset command required, but that shouldn't be necessary. */
static void
r8e_reset_8390(struct device *dev)
{
    int tmp = inb_p(NE_BASE + NE_RESET);
    int reset_start_time = jiffies;

    if (ei_debug > 1) printk("resetting the 8390 t=%ld...", jiffies);
    ei_status.txing = 0;

    outb_p(tmp, NE_BASE + NE_RESET);
    /* This check _should_not_ be necessary, omit eventually. */
    while ((inb_p(NE_BASE+EN0_ISR) & ENISR_RESET) == 0)
	if (jiffies - reset_start_time > 2) {
	    printk("%s: r8e_reset_8390() did not complete.\n", dev->name);
	    break;
	}
}

/* Block input and output, similar to the Crynwr packet driver.  If you
   porting to a new ethercard look at the packet driver source for hints.
   The NEx000 doesn't share it on-board packet memory -- you have to put
   the packet out through the "remote DMA" dataport using outb. */

static int
r8e_block_input(struct device *dev, int count, char *buf, int ring_offset)
{
#ifdef CONFIG_NE_SANITY
    int xfer_count = count;
#endif
    int nic_base = dev->base_addr;

    /* This *shouldn't* happen. If it does, it's the last thing you'll see */
    if (set_bit(0,(void*)&ei_status.dmaing)) {
	if (ei_debug > 0)
	    printk("%s: DMAing conflict in ne_block_input "
		   "[DMAstat:%d][irqlock:%d][intr:%d].\n",
		   dev->name, ei_status.dmaing, ei_status.irqlock,
		   dev->interrupt);
	return 0;
    }
    ei_status.dmaing |= 0x02;
    outb_p(E8390_NODMA+E8390_PAGE0+E8390_START, nic_base+ NE_CMD);
    outb_p(count & 0xff, nic_base + EN0_RCNTLO);
    outb_p(count >> 8, nic_base + EN0_RCNTHI);
    outb_p(ring_offset & 0xff, nic_base + EN0_RSARLO);
    outb_p(ring_offset >> 8, nic_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, nic_base + NE_CMD);
    if (ei_status.word16) {
      insw(NE_BASE + NE_DATAPORT,buf,count>>1);
      if (count & 0x01) {
	buf[count-1] = inb(NE_BASE + NE_DATAPORT);
#ifdef CONFIG_NE_SANITY
	xfer_count++;
#endif
      }
    } else {
	insb(NE_BASE + NE_DATAPORT, buf, count);
    }

    /* This was for the ALPHA version only, but enough people have
       been encountering problems so it is still here.  If you see
       this message you either 1) have a slightly incompatible clone
       or 2) have noise/speed problems with your bus. */
#ifdef CONFIG_NE_SANITY
    if (ei_debug > 1) {		/* DMA termination address check... */
	int addr, tries = 20;
	do {
	    /* DON'T check for 'inb_p(EN0_ISR) & ENISR_RDC' here
	       -- it's broken for Rx on some cards! */
	    int high = inb_p(nic_base + EN0_RSARHI);
	    int low = inb_p(nic_base + EN0_RSARLO);
	    addr = (high << 8) + low;
	    if (((ring_offset + xfer_count) & 0xff) == low)
		break;
	} while (--tries > 0);
	if (tries <= 0)
	    printk("%s: RX transfer address mismatch,"
		   "%#4.4x (expected) vs. %#4.4x (actual).\n",
		   dev->name, ring_offset + xfer_count, addr);
    }
#endif
    outb_p(ENISR_RDC, nic_base + EN0_ISR);	/* Ack intr. */
    ei_status.dmaing &= ~0x03;
    return ring_offset + count;
}

static void
r8e_block_output(struct device *dev, int count,
		const unsigned char *buf, const int start_page)
{
#ifdef CONFIG_NE_SANITY
    int retries = 0;
#endif
    int nic_base = NE_BASE;
    unsigned long dma_start;

    /* Round the count up for word writes.  Do we need to do this?
       What effect will an odd byte count have on the 8390?
       I should check someday. */
    if (ei_status.word16 && (count & 0x01))
      count++;

    /* This *shouldn't* happen. If it does, it's the last thing you'll see */
    if (set_bit(0,(void*)&ei_status.dmaing)) {
	if (ei_debug > 0)
	    printk("%s: DMAing conflict in ne_block_output."
		   "[DMAstat:%d][irqlock:%d][intr:%d]\n",
		   dev->name, ei_status.dmaing, ei_status.irqlock,
		   dev->interrupt);
	return;
    }
    ei_status.dmaing |= 0x04;
    /* We should already be in page 0, but to be safe... */
    outb_p(E8390_PAGE0+E8390_START+E8390_NODMA, nic_base + NE_CMD);

#ifdef CONFIG_NE_SANITY
 retry:
#endif

#ifdef CONFIG_NE_RW_BUGFIX 
    /* Handle the read-before-write bug the same way as the
       Crynwr packet driver -- the NatSemi method doesn't work.
       Actually this doesn't always work either, but if you have
       problems with your NEx000 this is better than nothing! */
    outb_p(0x42, nic_base + EN0_RCNTLO);
    outb_p(0x00,   nic_base + EN0_RCNTHI);
    outb_p(0x42, nic_base + EN0_RSARLO);
    outb_p(0x00, nic_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, nic_base + NE_CMD);
    /* Make certain that the dummy read has occurred. */
    SLOW_DOWN_IO;
    SLOW_DOWN_IO;
    SLOW_DOWN_IO;
#endif  /* rw_bugfix */

    outb_p(ENISR_RDC, nic_base + EN0_ISR);

   /* Now the normal output. */
    outb_p(count & 0xff, nic_base + EN0_RCNTLO);
    outb_p(count >> 8,   nic_base + EN0_RCNTHI);
    outb_p(0x00, nic_base + EN0_RSARLO);
    outb_p(start_page, nic_base + EN0_RSARHI);

    outb_p(E8390_RWRITE+E8390_START, nic_base + NE_CMD);
    if (ei_status.word16) {
	outsw(NE_BASE + NE_DATAPORT, buf, count>>1);
    } else {
	outsb(NE_BASE + NE_DATAPORT, buf, count);
    }

    dma_start = jiffies;

#ifdef CONFIG_NE_SANITY
    /* This was for the ALPHA version only, but enough people have
       been encountering problems so it is still here. */
    if (ei_debug > 1) {		/* DMA termination address check... */
	int addr, tries = 20;
	do {
	    int high = inb_p(nic_base + EN0_RSARHI);
	    int low = inb_p(nic_base + EN0_RSARLO);
	    addr = (high << 8) + low;
	    if ((start_page << 8) + count == addr)
		break;
	} while (--tries > 0);
	if (tries <= 0) {
	    printk("%s: Tx packet transfer address mismatch,"
		   "%#4.4x (expected) vs. %#4.4x (actual).\n",
		   dev->name, (start_page << 8) + count, addr);
	    if (retries++ == 0)
		goto retry;
	}
    }
#endif

    while ((inb_p(nic_base + EN0_ISR) & ENISR_RDC) == 0)
	if (jiffies - dma_start > NE_RDC_TIMEOUT) {
		printk("%s: timeout waiting for Tx RDC.\n", dev->name);
		r8e_reset_8390(dev);
		NS8390_init(dev,1);
		break;
	}

    outb_p(ENISR_RDC, nic_base + EN0_ISR);	/* Ack intr. */
    ei_status.dmaing &= ~0x05;
    return;
}


/*
 * Local variables:
 *  compile-command: "gcc -DKERNEL -Wall -O6 -fomit-frame-pointer -I/usr/src/linux/net/tcp -c rtl8019.c"
 *  version-control: t
 *  kept-new-versions: 5
 * End:
 */
