// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright IBM Corp. 2025
 *
 * Pseudo phmac (hmac with protected key) in-kernel crypto implementation
 * In fact this is a simple asynch wrapper around the synchronous shash
 * hmac(sha256). All update, final, finup and digest calls are always
 * postponed to an instance of the crypto_engine and thus executed asynchronous.
 */

#define KMSG_COMPONENT	"pseudo_phmac"
#define pr_fmt(fmt)	KMSG_COMPONENT ": " fmt

#include <crypto/engine.h>
#include <crypto/hash.h>
#include <crypto/internal/hash.h>
#include <crypto/sha2.h>
#include <linux/miscdevice.h>
#include <linux/module.h>

/*
 * If this define is active, then after selftest the setkey()
 * function assumes there is always a (s390 specific) 'clear
 * key token' given.
 * If this define is not active, this phmac behaves exactly
 * like a hmac algorithm.
 */
//#define ALWAYS_KEYTOKEN_AFTER_SELFTEST

static struct crypto_engine *phmac_crypto_engine;
#define MAX_QLEN 10

struct phmac_tfm_ctx {
	struct crypto_shash *shash_tfm;
};

struct phmac_state {
	char _desc[sizeof(struct shash_desc) + HASH_MAX_DESCSIZE];
};

enum async_op {
	OP_NOP = 0,
	OP_INIT,
	OP_UPDATE,
	OP_FINAL,
	OP_FINUP,
	OP_DIGEST
};

struct phmac_req_ctx {
	struct phmac_state state;
	int async_op;
};

static int do_init(struct ahash_request *req)
{
	struct crypto_ahash *tfm = crypto_ahash_reqtfm(req);
	struct phmac_tfm_ctx *tfm_ctx = crypto_ahash_ctx(tfm);
	struct crypto_shash *shash = tfm_ctx->shash_tfm;
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	struct phmac_state *state = &req_ctx->state;
	struct shash_desc *desc = (struct shash_desc *)state->_desc;
	int rc;

	pr_debug("req=%p req_ctx=%p state=%p\n", req, req_ctx, state);

	/*
	 * clear the request context and prep the shash desc
	 * which is stored in the request context.
	 */

	memset(req_ctx, 0, sizeof(*req_ctx));

	desc->tfm = shash;

	rc = crypto_shash_init(desc);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int do_update(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	struct phmac_state *state = &req_ctx->state;
	struct shash_desc *desc = (struct shash_desc *)state->_desc;
	struct crypto_hash_walk walk;
	int n, rc = 0;

	pr_debug("req=%p req_ctx=%p state=%p\n", req, req_ctx, state);

	n = crypto_hash_walk_first(req, &walk);
	while (n) {
		if (n < 0) {
			rc = n;
			crypto_hash_walk_done(&walk, rc);
		}
		if (n <= 0)
			break;
		rc = crypto_shash_update(desc, walk.data, n);
		if (rc) {
			crypto_hash_walk_done(&walk, rc);
			break;
		}
		n = crypto_hash_walk_done(&walk, 0);
	}

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int do_final(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	struct phmac_state *state = &req_ctx->state;
	struct shash_desc *desc = (struct shash_desc *)state->_desc;
	int rc;

	pr_debug("req=%p req_ctx=%p state=%p\n", req, req_ctx, state);

	rc = crypto_shash_final(desc, req->result);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int do_finup(struct ahash_request *req)
{
	int rc;

	pr_debug("req=%p\n", req);

	rc = do_update(req);
	if (rc)
		goto out;
	rc = do_final(req);

out:
	pr_debug("rc=%d\n", rc);
	return rc;
}

static int do_digest(struct ahash_request *req)
{
	int rc;

	pr_debug("req=%p\n", req);

	rc = do_init(req);
	if (rc)
		goto out;
	rc = do_update(req);
	if (rc)
		goto out;
	rc = do_final(req);

out:
	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_init(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("req=%p req_ctx=%p\n", req, req_ctx);

	/*
	 * init is in this implementation always synchronous.
	 * so just call the real function.
	 */

	rc = do_init(req);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_update(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("req=%p req_ctx=%p\n", req, req_ctx);

	/*
	 * update is in this implementation always asynchronous:
	 * prep the req for asynch processing and queue it into
	 * our engine instance.
	 */

	req_ctx->async_op = OP_UPDATE;
	rc = crypto_transfer_hash_request_to_engine(phmac_crypto_engine, req);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_final(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("req=%p req_ctx=%p\n", req, req_ctx);

	/*
	 * final is in this implementation always asynchronous:
	 * prep the req for asynch processing and queue it into
	 * our engine instance.
	 */

	req_ctx->async_op = OP_FINAL;
	rc = crypto_transfer_hash_request_to_engine(phmac_crypto_engine, req);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_finup(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("req=%p req_ctx=%p\n", req, req_ctx);

	/*
	 * finup is in this implementation always asynchronous:
	 * prep the req for asynch processing and queue it into
	 * our engine instance.
	 */

	req_ctx->async_op = OP_FINUP;
	rc = crypto_transfer_hash_request_to_engine(phmac_crypto_engine, req);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_digest(struct ahash_request *req)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("req=%p req_ctx=%p\n", req, req_ctx);

	/*
	 * digest is in this implementation always asynchronous:
	 * prep the req for asynch processing and queue it into
	 * our engine instance.
	 */

	req_ctx->async_op = OP_DIGEST;
	rc = crypto_transfer_hash_request_to_engine(phmac_crypto_engine, req);

	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_setkey(struct crypto_ahash *tfm,
			const u8 *key, unsigned int keylen)
{
	struct phmac_tfm_ctx *tfm_ctx = crypto_ahash_ctx(tfm);
	struct crypto_shash *shash = tfm_ctx->shash_tfm;
	unsigned int the_keylen;
	const u8 *the_key;
	int rc = -EINVAL;

	pr_debug("tfm=%p key=%p keylen=%u\n", tfm, key, keylen);

#ifdef ALWAYS_KEYTOKEN_AFTER_SELFTEST
	if (!crypto_ahash_tested(tfm)) {
		/* selftest running: key is a raw hmac clear key */
		the_key = key;
		the_keylen = keylen;
	} else {
		/*
		 * This is a key 'token'. Only hmac_clrkey_token is supported
		 * here. So check and extract the real raw key from this struct.
		 */
		struct hmac_clrkey_token {
			u8  type;
			u8  res0[3];
			u8  version;
			u8  res1[3];
			u32 keytype;
			u32 len;
			u8 key[];
		} __packed * t = (struct hmac_clrkey_token *)key;

		if (t->type != 0x00 || t->version != 0x02) {
			pr_debug("key token type 0x%02x or version 0x%02x mismatch\n",
				 t->type, t->version);
			goto out;
		}
		if (t->keytype != 12 /* this is PKEY_KEYTYPE_HMAC_512 */) {
			pr_debug("key token keytype 0x%04x mismatch\n", t->keytype);
			goto out;
		}
		the_key = t->key;
		the_keylen = t->len;
	}
#else
	/* always assume we get a raw hmac key */
	the_key = key;
	the_keylen = keylen;
#endif

	rc = crypto_shash_setkey(shash, the_key, the_keylen);

#ifdef ALWAYS_KEYTOKEN_AFTER_SELFTEST
out:
#endif
	pr_debug("rc=%d\n", rc);
	return rc;
}

static int phmac_export(struct ahash_request *req, void *out)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	struct phmac_state *state = &req_ctx->state;

	pr_debug("req=%p req_ctx=%p state=%p\n", req, req_ctx, state);

	memcpy(out, state, sizeof(*state));

	return 0;
}

static int phmac_import(struct ahash_request *req, const void *in)
{
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	struct phmac_state *state = &req_ctx->state;

	pr_debug("req=%p req_ctx=%p state=%p\n", req, req_ctx, state);

	memset(req_ctx, 0, sizeof(*req_ctx));
	memcpy(state, in, sizeof(*state));

	return 0;
}

static int phmac_init_tfm(struct crypto_ahash *tfm)
{
	struct phmac_tfm_ctx *tfm_ctx = crypto_ahash_ctx(tfm);
	struct crypto_shash *shash;
	int rc = 0;

	pr_debug("tfm=%p\n", tfm);

	memset(tfm_ctx, 0, sizeof(*tfm_ctx));
	crypto_ahash_set_reqsize(tfm, sizeof(struct phmac_req_ctx));

	shash = crypto_alloc_shash("hmac(sha256)", 0, 0);
	if (IS_ERR(shash)) {
		rc = PTR_ERR(shash);
		goto out;
	}
	tfm_ctx->shash_tfm = shash;

out:
	pr_debug("rc=%d\n", rc);
	return rc;
}

static void phmac_exit_tfm(struct crypto_ahash *tfm)
{
	struct phmac_tfm_ctx *tfm_ctx = crypto_ahash_ctx(tfm);

	pr_debug("tfm=%p\n", tfm);

	if (tfm_ctx->shash_tfm)
		crypto_free_shash(tfm_ctx->shash_tfm);
}

static int phmac_do_one_request(struct crypto_engine *engine, void *areq)
{
	struct ahash_request *req = ahash_request_cast(areq);
	struct phmac_req_ctx *req_ctx = ahash_request_ctx(req);
	int rc;

	pr_debug("areq=%p\n", areq);

	switch (req_ctx->async_op) {
	case OP_INIT:
		rc = do_init(req);
		break;
	case OP_UPDATE:
		rc = do_update(req);
		break;
	case OP_FINAL:
		rc = do_final(req);
		break;
	case OP_FINUP:
		rc = do_finup(req);
		break;
	case OP_DIGEST:
		rc = do_digest(req);
		break;
	default:
		rc = -EINVAL;
	}

	pr_debug("request complete with rc=%d\n", rc);
	local_bh_disable();
	crypto_finalize_hash_request(engine, req, rc);
	local_bh_enable();

	return rc;
}

static struct {
	struct ahash_engine_alg alg;
	bool registered;
} phmac_alg = {
	.alg = {
		.base = {
			.init	  = phmac_init,
			.update	  = phmac_update,
			.final	  = phmac_final,
			.finup	  = phmac_finup,
			.digest	  = phmac_digest,
			.setkey	  = phmac_setkey,
			.import	  = phmac_import,
			.export	  = phmac_export,
			.init_tfm = phmac_init_tfm,
			.exit_tfm = phmac_exit_tfm,
			.halg = {
				.digestsize = SHA256_DIGEST_SIZE,
				.statesize  = sizeof(struct phmac_state),
				.base = {
					.cra_name = "phmac(sha256)",
					.cra_driver_name = "phmac_sha256",
					.cra_blocksize = SHA256_BLOCK_SIZE,
					.cra_priority = 400,
					.cra_flags = CRYPTO_ALG_ASYNC |
						CRYPTO_ALG_NO_FALLBACK,
					.cra_ctxsize = sizeof(struct phmac_tfm_ctx),
					.cra_module = THIS_MODULE,
				},
			},
		},
		.op = {
			.do_one_request = phmac_do_one_request,
		},
	}
};

static struct miscdevice phmac_dev = {
	.name	= "pseudo_phmac",
	.minor	= MISC_DYNAMIC_MINOR,
};

static void pseudo_phmac_exit(void)
{
	if (phmac_crypto_engine) {
		crypto_engine_stop(phmac_crypto_engine);
		crypto_engine_exit(phmac_crypto_engine);
	}

	if (phmac_alg.registered)
		crypto_engine_unregister_ahash(&phmac_alg.alg);

	misc_deregister(&phmac_dev);
}

static int __init pseudo_phmac_init(void)
{
	int rc;

	rc = misc_register(&phmac_dev);
	if (rc)
		return rc;

	phmac_crypto_engine =
		crypto_engine_alloc_init_and_set(phmac_dev.this_device,
						 true, false, MAX_QLEN);
	if (!phmac_crypto_engine) {
		rc = -ENOMEM;
		goto out_err;
	}
	rc = crypto_engine_start(phmac_crypto_engine);
	if (rc) {
		crypto_engine_exit(phmac_crypto_engine);
		phmac_crypto_engine = NULL;
		goto out_err;
	}

	rc = crypto_engine_register_ahash(&phmac_alg.alg);
	if (rc)
		goto out_err;
	phmac_alg.registered = true;
	pr_debug("%s registered\n", phmac_alg.alg.base.halg.base.cra_name);

	return 0;

out_err:
	pseudo_phmac_exit();
	return rc;
}

module_init(pseudo_phmac_init);
module_exit(pseudo_phmac_exit);

MODULE_ALIAS_CRYPTO("phmac(sha256)");

MODULE_DESCRIPTION("Pseudo protected key HMAC driver");
MODULE_LICENSE("GPL");
