/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Ant" and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs;


import java.util.regex.*;
import java.util.*;

import org.apache.tools.ant.types.*;
import org.apache.tools.ant.filters.*;
import org.apache.tools.ant.filters.util.ChainReaderHelper;
import org.apache.tools.ant.util.regexp.*;
import org.apache.tools.ant.util.*;
import org.apache.tools.ant.*;
import org.apache.tools.ant.taskdefs.*;
import java.io.*;


/**
 * This class contains the 'concat' task, used to concatenate a series
 * of files into a single stream. The destination of this stream may
 * be the system console, or a file. The following is a sample
 * invocation:
 *
 * <pre>
 * &lt;concat destfile=&quot;${build.dir}/index.xml&quot;
 *   append=&quot;false&quot;&gt;
 *
 *   &lt;fileset dir=&quot;${xml.root.dir}&quot;
 *     includes=&quot;*.xml&quot; /&gt;
 *
 * &lt;/concat&gt;
 * </pre>
 *
 * @author <a href="mailto:derek@activate.net">Derek Slager</a>
 * @author Peter Reilly
 */
public class Concat
    extends Task
{
    private Vector        sources = new Vector();
    private Vector        filterChains = null;
    private File          destFile = null;
    private boolean       forceOverwrite = true;
    private TextElement   footer;
    private TextElement   header;
    private String        encoding = null;
    private boolean       append = false;
    private StringBuffer  textBuffer = null;
    private Vector        sourceFiles = new Vector();
    private FileUtils     fileUtils = FileUtils.newFileUtils();
    
    /**
     * Sets the destination file, or uses the console if not specified.
     */
    public void setDestfile(File destFile) {
        this.destFile = destFile;
    }


    /**
     * Sets the behavior when the destination file exists. If set to
     * <code>true</code> the stream data will be appended to the
     * existing file, otherwise the existing file will be
     * overwritten. Defaults to <code>false</code>.
     */
    public void setAppend(boolean append) {
        this.append = append;
    }

    /**
     * Sets the character encoding
     */
    public void setEncoding (String encoding) {
        this.encoding = encoding;
    }


    /**
     * Path of files to concatenate.
     */
     public Path createPath() {
        Path path = new Path(getProject());
        sources.addElement(path);
        return path;
    }

    /**
     * Set of files to concatenate.
     */
    public void addFileset(FileSet set) {
        sources.addElement(set);
    }
    
    /**
     * List of files to concatenate.
     */
    public void addFilelist(FileList list) {
        sources.addElement(list);
    }
    
    /**
     * This method adds text which appears in the 'concat' element.
     */
    public void addText(String text) {
        if (textBuffer == null) {
            textBuffer = new StringBuffer(text.length());
        }
            
        textBuffer.append(text);
    }

    /**
     * Adds a FilterChain.
     */
    public void addFilterChain(FilterChain filterChain) {
        if (filterChains == null)
            filterChains = new Vector();
        filterChains.addElement(filterChain);
    }

    /**
     * Overwrite existing destination file
     */
    public void setOverwrite(boolean overwrite) {
        this.forceOverwrite = overwrite;
    }

    /**
     * Add a header to the concatenated output
     */
    public void addHeader(TextElement el) {
        this.header = el;
    }

    /**
     * Add a footer to the concatenated output
     */
    public void addFooter(TextElement el) {
        this.footer = el;
    }

    /**
     * This method performs the concatenation.
     */
    public void execute() {
        // treat empty nested text as no text
        sanitizeText();
        
        // Sanity check our inputs.
        if (sources.size() == 0 && textBuffer == null) {
            // Nothing to concatenate!
            throw new BuildException("At least one file " + 
                                     "must be provided, or " + 
                                     "some text.");
        }
        
        // If using filesets, disallow inline text. This is similar to
        // using GNU 'cat' with file arguments -- stdin is simply
        // ignored.
        if (sources.size() > 0 && textBuffer != null) {
            throw new BuildException("Cannot include inline text " + 
                                     "when using filesets.");
        }

        // Iterate thru the sources - paths, filesets and filelists
        for (Enumeration e = sources.elements(); e.hasMoreElements();) {
            Object o = e.nextElement();
            if (o instanceof Path) {
                Path path = (Path) o;
                checkAddFiles(null, path.list());
            }
            else if (o instanceof FileSet) {
                FileSet fileSet = (FileSet) o;
                DirectoryScanner scanner =
                    fileSet.getDirectoryScanner(getProject());
                checkAddFiles(fileSet.getDir(getProject()),
                              scanner.getIncludedFiles());
            }
            else if (o instanceof FileList) {
                FileList fileList = (FileList) o;
                checkAddFiles(fileList.getDir(getProject()),
                              fileList.getFiles(getProject()));
            }
        }
        // check if the files are outofdate
        if (destFile != null
            && !forceOverwrite
            && (sourceFiles.size() > 0)
            && destFile.exists())
        {
            boolean outofdate = false;
            for (int i = 0; i < sourceFiles.size(); ++i) {
                File file = (File) sourceFiles.elementAt(i);
                if (file.lastModified() > destFile.lastModified())
                {
                    outofdate = true;
                    break;
                }
            }
            if (! outofdate)
                return; // no need to do anything
        }
        // Do nothing if all the sources are not present
        // And textBuffer is null
        if (textBuffer == null && sourceFiles.size() == 0) {
            log("No existing files, doing nothing", Project.MSG_WARN);
            return;
        }
        catFiles();
    }

    private void checkAddFiles(File base, String[] filenames) {
        for (int i = 0; i < filenames.length; ++i) {
            File file = new File(base, filenames[i]);
            if (! file.exists()) {
                log("File " + file + " does not exist.", Project.MSG_ERR);
                continue;
            }
            if (destFile != null &&
                destFile.getAbsolutePath().equals(
                    file.getAbsolutePath()))
            {
                throw new BuildException(
                    "Input file \"" +
                    file.getName() + "\" is the same as the output file.");
            }
            sourceFiles.addElement(file);
        }
    }
    

    /** perform the concatenation */
    private void catFiles() {
        OutputStream os = null;
        InputStream  is = null;
        char[]       buffer = new char[8192];

        try {

            if (destFile == null) {
                // Log using WARN so it displays in 'quiet' mode.
                os = new LogOutputStream(this, Project.MSG_WARN);
            } else {
                // ensure that the parent dir of dest file exists
                File parent = fileUtils.getParentFile(destFile);
                if (! parent.exists())
                    parent.mkdirs();
                
                os =
                    new FileOutputStream(destFile.getAbsolutePath(), append);

            }

            PrintWriter writer = null;
            if (encoding == null) {
                writer = new PrintWriter(
                    new BufferedWriter(
                        new OutputStreamWriter(os)));
            }
            else {
                writer = new PrintWriter(
                    new BufferedWriter(
                        new OutputStreamWriter(os,encoding)));
            }


            if (header != null) {
                writer.println(header.getValue());
            }

            if (textBuffer != null) {
                writer.write(getProject().replaceProperties(
                                 textBuffer.toString()));
            }

            for (int i = 0; i < sourceFiles.size(); ++i) {
                File file = (File) sourceFiles.elementAt(i);
                is = new FileInputStream(file);
                BufferedReader in = null;
                if (encoding == null) {
                    in = new BufferedReader(new InputStreamReader(is));
                }
                else {
                    in = new BufferedReader(
                        new InputStreamReader(is, encoding));
                }
                if (filterChains != null) {
                    ChainReaderHelper helper = new ChainReaderHelper();
                    helper.setBufferSize(8192);
                    helper.setPrimaryReader(in);
                    helper.setFilterChains(filterChains);
                    helper.setProject(getProject());
                    in = new BufferedReader(helper.getAssembledReader());
                }

                while (true) {
                    int nRead = in.read(buffer, 0, buffer.length);
                    if (nRead == -1)
                        break;
                    writer.write(buffer, 0, nRead);
                }
                
                writer.flush();
                os.flush();
                is.close();
                is = null;
            }

            if (footer != null) {
                writer.println(footer.getValue());
            }
            writer.flush();
            os.flush();

        } catch (IOException ioex) {
            throw new BuildException("Error while concatenating: "
                                     + ioex.getMessage(), ioex);
        }
        catch (Throwable t) {
            throw new BuildException("Error while concatenating: ", t);
        }
        finally {
            try {is.close();} catch (Throwable ignore) {}
            try {os.close();} catch (Throwable ignore) {}
        }
    }


    /**
     * sub element points to a file or contains text
     */

    public static class TextElement {
        private FileUtils    fileUtils = FileUtils.newFileUtils();
        private String   value;
        private boolean  trimLeading = false;
        private boolean  trim = false;

        public void setFile(File file) {
            // Ignore non existing files
            if (! file.exists())
                return;
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(
                    new FileReader(file));
                value = fileUtils.readFully(reader);
            }
            catch (IOException ex) {
                throw new BuildException(ex);
            }
            finally {
                try {reader.close();} catch (Throwable t) {}
            }
        }
        
        public void addText(String value) {
            if (value.trim().length() == 0)
                return;
            this.value = value;
        }
        public void setTrimLeading(boolean strip) {
            this.trimLeading = strip;
        }
        public void setTrim(boolean trim) {
            this.trim = trim;
        }
        public String getValue() {
            if (value == null)
                value = "";
            if (value.trim().length() == 0)
                value = "";
            if (trimLeading) {
                char[] current = value.toCharArray();
                StringBuffer b = new StringBuffer(current.length);
                boolean startOfLine = true;
                int pos = 0;
                while (pos < current.length) {
                    char ch = current[pos++];
                    if (startOfLine) {
                        if (ch == ' ' || ch == '\t')
                            continue;
                        startOfLine = false;
                    }
                    b.append(ch);
                    if (ch == '\n' || ch == '\r')
                        startOfLine = true;
                }
                value = b.toString();
            }
            if (trim)
                value = value.trim();
            return value;
        }
    }
    /**
     * Treat empty nested text as no text.
     *
     * <p>Depending on the XML parser, addText may have been called
     * for &quot;ignorable whitespace&quot; as well.</p>
     */
    private void sanitizeText() {
        if (textBuffer != null) {
            if (textBuffer.toString().trim().length() == 0) {
                textBuffer = null;
            }
        }
    }
 }

